/***************************************************************************

  Capcom System 2
  ===============


    Driver by Paul Leaman (paul@vortexcomputing.demon.co.uk)

    Thanks to Raz, Crashtest and the CPS2 decryption team without whom
    none of this would have been possible.

***************************************************************************/

#include "driver.h"
#include "machine/eeprom.h"
#include "cpu/m68000/m68000.h"

#include "cps1.h"       /* External CPS1 definitions */
#ifdef MAME32JP
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

/*
Export this function so that the vidhrdw routine can drive the
Q-Sound hardware
*/
WRITE16_HANDLER( cps2_qsound_sharedram_w )
{
    qsound_sharedram1_w(offset/2, data, 0xff00);
}

/* Maximum size of Q Sound Z80 region */
#define QSOUND_SIZE 0x50000

/* Maximum 680000 code size */
#undef  CODE_SIZE
#define CODE_SIZE   0x0400000


extern data16_t *cps2_objram1,*cps2_objram2;
extern data16_t *cps2_output;
extern size_t cps2_output_size;
extern VIDEO_START( cps2 );

extern int scanline1;
extern int scanline2;




static int readpaddle;

/**********************************************************************************/

#ifdef MAME32JP
static data16_t *cps2_ram;

static int region_switch_offset;
static int region_switch_data;
static int region_switch_mask;
static int region_switch_count;
static int region_switch_counter;

static int region_switch_offset2;
static int region_switch_data2;
static int region_switch_mask2;
static int region_switch_count2;
static int region_switch_counter2;

READ16_HANDLER( cps2_region_16_r )
{
	data16_t mem = cps2_ram[region_switch_offset >> 1];

	if (region_switch_counter < region_switch_count)
	{
		region_switch_counter++;
		return mem;
	}
	return (mem & region_switch_mask) | region_switch_data;
}

READ16_HANDLER( cps2_region_16_r2 )
{
	data16_t mem = cps2_ram[region_switch_offset2 >> 1];

	if (region_switch_counter2 < region_switch_count2)
	{
		region_switch_counter2++;
		return mem;
	}
	return (mem & region_switch_mask2) | region_switch_data2;
}

#define MASK_HIGH  0xff00
#define MASK_LOW   0x00ff
#define MASK_NONE  0x0000

#define CPS2_REGION_16_R(NAME, OFFSET, DATA, COUNT, MASK) \
if (!strcmp(Machine->gamedrv->name, NAME))                \
{                                                         \
    region_switch_offset  = OFFSET & 0xffff;              \
    region_switch_data    = DATA;                         \
    region_switch_mask    = MASK;                         \
    region_switch_count   = COUNT;                        \
    if (region_switch_mask == MASK_LOW)                   \
        region_switch_data <<= 8;                         \
    install_mem_read16_handler(0, OFFSET, OFFSET+1, cps2_region_16_r); \
}

#define CPS2_REGION_16_R2(NAME, OFFSET, DATA, COUNT, MASK) \
if (!strcmp(Machine->gamedrv->name, NAME))                 \
{                                                          \
    region_switch_offset2  = OFFSET & 0xffff;              \
    region_switch_data2    = DATA;                         \
    region_switch_mask2    = MASK;                         \
    region_switch_count2   = COUNT;                        \
    if (region_switch_mask2 == MASK_LOW)                   \
        region_switch_data2 <<= 8;                         \
    install_mem_read16_handler(0, OFFSET, OFFSET+1, cps2_region_16_r2); \
}
#endif

void init_cps2_driver(void)
{
#ifdef MAME32JP
	region_switch_counter  = 0;
	region_switch_counter2 = 0;

	CPS2_REGION_16_R("19xxregion",     0xff4d6e, 0, 4, MASK_LOW)
	CPS2_REGION_16_R("batcirjregion",  0xff58c4, 1, 4, MASK_LOW)
	CPS2_REGION_16_R("ecofghtrregion", 0xff03e8, 0, 4, MASK_HIGH)
	CPS2_REGION_16_R("xmvsfregion",    0xff5478, 0, 4, MASK_HIGH)
#endif

}

/**********************************************************************************/

static INTERRUPT_GEN( cps2_interrupt )
{
	static int scancount;

	/* 2 is vblank, 4 is some sort of scanline interrupt, 6 is both at the same time. */

	if(scancount >= 261)
	{
		scancount = -1;

	}
	scancount++;

	if(cps1_output[0x50/2] & 0x8000)
		cps1_output[0x50/2] = cps1_output[0x50/2] & 0x1ff;
	if(cps1_output[0x52/2] & 0x8000)
		cps1_output[0x52/2] = cps1_output[0x52/2] & 0x1ff;

//	usrintf_showmessage("%04x %04x - %04x %04x",scanline1,scanline2,cps1_output[0x50/2],cps1_output[0x52/2]);

	/* raster effects */

	{
		cps1_output[0x50/2] = 0;
		if(scanline1) cpu_set_irq_line(0, 4, HOLD_LINE);

		force_partial_update(16 - 10 + scancount);	/* Machine->visible_area.min_y - [first visible line?] + scancount */

//			usrintf_showmessage("IRQ4 scancounter = %04i",scancount);
	}

	/* raster effects */

	{
		cps1_output[0x52/2] = 0;
		if(scanline2) cpu_set_irq_line(0, 4, HOLD_LINE);

		force_partial_update(16 - 10 + scancount);	/* Machine->visible_area.min_y - [first visible line?] + scancount */

//			usrintf_showmessage("IRQ4 scancounter = %04i",scancount);
	}

	if(scancount == 256)  /* VBlank */
	{
		cps1_output[0x50/2] = scanline1;
		cps1_output[0x52/2] = scanline2;
		cpu_set_irq_line(0, 2, HOLD_LINE);

		{

			force_partial_update(256);
		}

	}

}



static struct EEPROM_interface cps2_eeprom_interface =
{
	6,		/* address bits */
	16,		/* data bits */
	"0110",	/*  read command */
	"0101",	/* write command */
	"0111"	/* erase command */
};

static NVRAM_HANDLER( cps2 )
{
	if (read_or_write)
		EEPROM_save(file);
	else
	{
        EEPROM_init(&cps2_eeprom_interface);

		if (file)
			EEPROM_load(file);
		else
		{
			if(!strcmp(Machine->gamedrv->name,"sfa3p") ||
			!strcmp(Machine->gamedrv->name,"sfa3r1p") ||
			!strcmp(Machine->gamedrv->name,"sfz3ap") ||
			!strcmp(Machine->gamedrv->name,"sfz3jr1p") ||
			!strcmp(Machine->gamedrv->name,"sfz3jp"))
			{
				data8_t eeprom[] = {0x00,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
						    0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
						    0x98,0x06,0x29,0x00,0x53,0x5A,0x33,0x4A,0x20,0x5A,0x45,0x52,0x4F,0x20,0x33,0x20,
						    0x00,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
						    0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
						    0x98,0x06,0x29,0x00,0x53,0x5A,0x33,0x4A,0x20,0x5A,0x45,0x52,0x4F,0x20,0x33,0x20,
						    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
						    0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};  /* sfz3jr1 */

				if(!strcmp(Machine->gamedrv->name,"sfz3jp"))
				{
					eeprom[0x21] = eeprom[0x51] = 0x07;
					eeprom[0x22] = eeprom[0x52] = 0x27;
				} else if(!strcmp(Machine->gamedrv->name,"sfa3p")) {
					eeprom[0x21] = eeprom[0x51] = 0x09;
					eeprom[0x22] = eeprom[0x52] = 0x04;
				} else if(!strcmp(Machine->gamedrv->name,"sfa3r1p")) {
					eeprom[0x08] = eeprom[0x23] = eeprom[0x38] = eeprom[0x53] = 0x02;
					eeprom[0x06] = eeprom[0x36] = 0x01;
					eeprom[0x10] = eeprom[0x40] = 0x04;
					eeprom[0x27] = eeprom[0x57] = 'U';
					eeprom[0x28] = eeprom[0x58] = 'A';
					eeprom[0x29] = eeprom[0x59] = 'L';
					eeprom[0x2a] = eeprom[0x5a] = 'P';
					eeprom[0x2b] = eeprom[0x5b] = 'H';
					eeprom[0x2c] = eeprom[0x5c] = 'A';
				} else if(!strcmp(Machine->gamedrv->name,"sfz3ap")) {
					eeprom[0x08] = eeprom[0x38] = 0x02;
					eeprom[0x06] = eeprom[0x22] = eeprom[0x36] = eeprom[0x52] = 0x01;
					eeprom[0x10] = eeprom[0x21] = eeprom[0x40] = eeprom[0x51] = 0x07;
					eeprom[0x13] = eeprom[0x43] = 0x03;
					eeprom[0x23] = eeprom[0x53] = 0x08;
					eeprom[0x27] = eeprom[0x57] = 'A';
					eeprom[0x29] = eeprom[0x59] = 'z';
					eeprom[0x2a] = eeprom[0x5a] = 'e';
					eeprom[0x2b] = eeprom[0x5b] = 'r';
					eeprom[0x2c] = eeprom[0x5c] = 'o';
				}

				EEPROM_set_data(eeprom, 128);
			}
		}
	}
}

READ16_HANDLER( cps2_eeprom_port_r )
{
    return (input_port_2_word_r(offset,0) & 0xfffe) | EEPROM_read_bit();
}

WRITE16_HANDLER( cps2_eeprom_port_w )
{
    if (ACCESSING_MSB)
    {
	/* bit 0 - Unused */
	/* bit 1 - Unused */
	/* bit 2 - Unused */
	/* bit 3 - Unused? */
	/* bit 4 - Eeprom data  */
	/* bit 5 - Eeprom clock */
	/* bit 6 - */
	/* bit 7 - */

	/* EEPROM */
	EEPROM_write_bit(data & 0x1000);
	EEPROM_set_clock_line((data & 0x2000) ? ASSERT_LINE : CLEAR_LINE);
	EEPROM_set_cs_line((data & 0x4000) ? CLEAR_LINE : ASSERT_LINE);
	}

	if (ACCESSING_LSB)
	{
	/* bit 0 - coin counter 1 */
	/* bit 0 - coin counter 2 */
	/* bit 2 - Unused */
	/* bit 3 - Allows access to Z80 address space (Z80 reset) */
	/* bit 4 - lock 1  */
	/* bit 5 - lock 2  */
	/* bit 6 - */
	/* bit 7 - */

        /* Z80 Reset */
		cpu_set_reset_line(1,(data & 0x0008) ? CLEAR_LINE : ASSERT_LINE);

	coin_counter_w(0, data & 0x0001);
	if((strncmp(Machine->gamedrv->name,"pzloop2",8)==0) ||
		(strncmp(Machine->gamedrv->name,"pzloop2j",8)==0))
	{
		// Puzz Loop 2 uses coin counter 2 input to switch between stick and paddle controls
		readpaddle = data & 0x0002;
	}
	else
	{
		coin_counter_w(1, data & 0x0002);
	}

	if(!strcmp(Machine->gamedrv->name,"mmatrix") ||		// Mars Matrix seems to require the coin lockout bit to be reversed
	   !strcmp(Machine->gamedrv->name,"mmatrixj"))
	{
		coin_lockout_w(0,data & 0x0010);
		coin_lockout_w(1,data & 0x0020);
		coin_lockout_w(2,data & 0x0040);
		coin_lockout_w(3,data & 0x0080);
	}
	else
	{
		coin_lockout_w(0,~data & 0x0010);
		coin_lockout_w(1,~data & 0x0020);
		coin_lockout_w(2,~data & 0x0040);
		coin_lockout_w(3,~data & 0x0080);
	}

	/*
	set_led_status(0,data & 0x01);
	set_led_status(1,data & 0x10);
	set_led_status(2,data & 0x20);
	*/
    }
}


READ16_HANDLER( cps2_qsound_volume_r )
{
	/* Extra adapter memory (0x660000-0x663fff) available when bit 14 = 0 */
	/* Network adapter (ssf2tb) present when bit 15 = 0 */
	/* Only game known to use both these so far is SSF2TB */

const char *gamename = Machine->gamedrv->name;
	if(strcmp(gamename,"ssf2tb")==0)
		return 0x2021;
	else
		return 0xe021;
}







static READ16_HANDLER( kludge_r )
{
	return 0xffff;
}


static MEMORY_READ16_START( cps2_readmem )
	{ 0x000000, 0x3fffff, MRA16_ROM },				/* 68000 ROM */
	{ 0x400000, 0x40000b, MRA16_RAM },				/* CPS2 object output */
	{ 0x618000, 0x619fff, qsound_sharedram1_r },		/* Q RAM */
	{ 0x662000, 0x662001, MRA16_RAM },				/* Network adapter related, accessed in SSF2TB */
	{ 0x662008, 0x662009, MRA16_RAM },				/* Network adapter related, accessed in SSF2TB */
	{ 0x662020, 0x662021, MRA16_RAM },				/* Network adapter related, accessed in SSF2TB */
	{ 0x660000, 0x663fff, MRA16_RAM },				/* When bit 14 of 0x804030 equals 0 this space is available. Many games store highscores and other info here if available. */
	{ 0x664000, 0x664001, MRA16_RAM },				/* Unknown - Only used if 0x660000-0x663fff available (could be RAM enable?) */
	{ 0x708000, 0x709fff, cps2_objram2_r },			/* Object RAM */
	{ 0x70a000, 0x70bfff, cps2_objram2_r },			/* mirror */
	{ 0x70c000, 0x70dfff, cps2_objram2_r },			/* mirror */
	{ 0x70e000, 0x70ffff, cps2_objram2_r },			/* mirror */
	{ 0x800100, 0x8001ff, cps1_output_r },			/* Output ports mirror (sfa) */
	{ 0x804000, 0x804001, input_port_0_word_r },		/* IN0 */
	{ 0x804010, 0x804011, input_port_1_word_r },		/* IN1 */
	{ 0x804020, 0x804021, cps2_eeprom_port_r  },		/* IN2 + EEPROM */
	{ 0x804030, 0x804031, cps2_qsound_volume_r },		/* Master volume. Also when bit 14=0 addon memory is present, when bit 15=0 network adapter present. */
	{ 0x8040b0, 0x8040b3, kludge_r },  				/* unknown (xmcotaj hangs if this is 0) */
	{ 0x804100, 0x8041ff, cps1_output_r },			/* CPS1 Output ports */
	{ 0x900000, 0x92ffff, MRA16_RAM },				/* Video RAM */
	{ 0xff0000, 0xffffff, MRA16_RAM },				/* RAM */
MEMORY_END

static WRITE16_HANDLER( pip )
{
	usrintf_showmessage("%04x",data);
}

static MEMORY_WRITE16_START( cps2_writemem )
	{ 0x000000, 0x3fffff, MWA16_ROM },				/* ROM */
	{ 0x400000, 0x40000b, MWA16_RAM, &cps2_output, &cps2_output_size },	/* CPS2 output */
	{ 0x618000, 0x619fff, qsound_sharedram1_w },		/* Q RAM */
	{ 0x662000, 0x662001, MWA16_RAM },				/* Network adapter related, accessed in SSF2TB */
	{ 0x662008, 0x662009, MWA16_RAM },				/* Network adapter related, accessed in SSF2TB (not sure if this port is write only yet)*/
	{ 0x662020, 0x662021, MWA16_RAM },				/* Network adapter related, accessed in SSF2TB */
	{ 0x660000, 0x663fff, MWA16_RAM },				/* When bit 14 of 0x804030 equals 0 this space is available. Many games store highscores and other info here if available. */
	{ 0x664000, 0x664001, MWA16_RAM },				/* Unknown - Only used if 0x660000-0x663fff available (could be RAM enable?) */
	{ 0x700000, 0x701fff, cps2_objram1_w, &cps2_objram1 },	/* Object RAM, no game seems to use it directly */
	{ 0x708000, 0x709fff, cps2_objram2_w, &cps2_objram2 },	/* Object RAM */
	{ 0x70a000, 0x70bfff, cps2_objram2_w },			/* mirror */
	{ 0x70c000, 0x70dfff, cps2_objram2_w },			/* mirror */
	{ 0x70e000, 0x70ffff, cps2_objram2_w },			/* mirror */
	{ 0x800100, 0x8001ff, cps1_output_w },			/* Output ports mirror (sfa) */
	{ 0x804040, 0x804041, cps2_eeprom_port_w },		/* EEPROM */
	{ 0x8040a0, 0x8040a1, MWA16_NOP },				/* Unknown (reset once on startup) */
	{ 0x8040e0, 0x8040e1, cps2_objram_bank_w },		/* bit 0 = Object ram bank swap */
	{ 0x804100, 0x8041ff, cps1_output_w, &cps1_output, &cps1_output_size },  /* Output ports */
	{ 0x900000, 0x92ffff, cps1_gfxram_w, &cps1_gfxram, &cps1_gfxram_size },
#ifdef MAME32JP
	{ 0xff0000, 0xffffff, MWA16_RAM, &cps2_ram },	/* RAM */
#else
	{ 0xff0000, 0xffffff, MWA16_RAM },				/* RAM */
#endif
MEMORY_END




INPUT_PORTS_START( 19xx )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( cybots )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( ssf2 )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
#ifdef MAME32JP
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS1    | IPF_PLAYER1 )
    PORT_BIT( 0x0010+0x0020+0x0040, IP_ACTIVE_LOW, IPT_CPS2    | IPF_PLAYER1 )
#endif
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
#ifdef MAME32JP
    PORT_BIT( 0x1000+0x2000, IP_ACTIVE_LOW, IPT_CPS1    | IPF_PLAYER2 )
    PORT_BIT( 0x1000+0x2000+0x4000, IP_ACTIVE_LOW, IPT_CPS2    | IPF_PLAYER2 )
#endif
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER1 )
#ifdef MAME32JP
    PORT_BIT( 0x0001+0x0002, IP_ACTIVE_LOW, IPT_CPS3    | IPF_PLAYER1 )
    PORT_BIT( 0x0001+0x0002+0x0004, IP_ACTIVE_LOW, IPT_CPS4    | IPF_PLAYER1 )
#endif
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER2 )
#ifdef MAME32JP
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS3    | IPF_PLAYER2 )
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS4    | IPF_PLAYER2 )
#endif
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER2 )
#ifdef MAME32JP
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_CPS4    | IPF_PLAYER2 )
#endif
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( ddtod )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
#ifdef MAME32JP
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER1 )
#endif
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
#ifdef MAME32JP
    PORT_BIT( 0x1000+0x2000, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER2 )
#endif

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER3 )
#ifdef MAME32JP
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER3 )
#endif
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER4 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER4 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER4 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER4 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER4 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER4 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER4 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER4 )
#ifdef MAME32JP
    PORT_BIT( 0x1000+0x2000, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER4 )
#endif

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_START3 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_START4 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_COIN3 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_COIN4 )
INPUT_PORTS_END

INPUT_PORTS_START( avsp )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
#ifdef MAME32JP
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER1 )
    PORT_BIT( 0x0020+0x0040, IP_ACTIVE_LOW, IPT_CPS6    | IPF_PLAYER1 )
#endif
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
#ifdef MAME32JP
    PORT_BIT( 0x1000+0x2000, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER2 )
    PORT_BIT( 0x2000+0x4000, IP_ACTIVE_LOW, IPT_CPS6    | IPF_PLAYER2 )
#endif
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
#ifdef MAME32JP
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER3 )
    PORT_BIT( 0x0020+0x0040, IP_ACTIVE_LOW, IPT_CPS6    | IPF_PLAYER3 )
#endif
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_START3 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_COIN3 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( qndream )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( batcir )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
#ifdef MAME32JP
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER1 )
#endif
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
#ifdef MAME32JP
    PORT_BIT( 0x1000+0x2000, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER2 )
#endif
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER3 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
#ifdef MAME32JP
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER3 )
#endif
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER4 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER4 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER4 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER4 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER4 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER4 )
#ifdef MAME32JP
    PORT_BIT( 0x1000+0x2000, IP_ACTIVE_LOW, IPT_CPS5    | IPF_PLAYER4 )
#endif
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_START3 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_START4 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_COIN3 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_COIN4 )
INPUT_PORTS_END

INPUT_PORTS_START( sgemf )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
#ifdef MAME32JP
    PORT_BIT( 0x0010+0x0020, IP_ACTIVE_LOW, IPT_CPS5 | IPF_PLAYER1 )
    PORT_BIT( 0x0020+0x0040, IP_ACTIVE_LOW, IPT_CPS6 | IPF_PLAYER1 )
    PORT_BIT( 0x0010+0x0020+0x0040, IP_ACTIVE_LOW, IPT_CPS7 | IPF_PLAYER1 )
#endif
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
#ifdef MAME32JP
    PORT_BIT( 0x1000+0x2000, IP_ACTIVE_LOW, IPT_CPS5 | IPF_PLAYER2 )
    PORT_BIT( 0x2000+0x4000, IP_ACTIVE_LOW, IPT_CPS6 | IPF_PLAYER2 )
    PORT_BIT( 0x1000+0x2000+0x4000, IP_ACTIVE_LOW, IPT_CPS7 | IPF_PLAYER2 )
#endif
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( ringdest )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER2 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( cps2 )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER2 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER2 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( puzloop2 )
    PORT_START      /* IN0 (0x00) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )

    PORT_START      /* IN1 (0x10) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START      /* IN2 (0x20) */
    PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_SPECIAL )   /* EEPROM bit */
	PORT_BITX(0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
    PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
    PORT_BIT( 0x00f8, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN1 )
    PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN2 )
    PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_ANALOGX( 0x0fff, 0x0000, IPT_DIAL | IPF_PLAYER1, 100, 15, 0, 0, KEYCODE_Z, KEYCODE_X, 0, 0 )

	PORT_START
	PORT_ANALOGX( 0x0fff, 0x0000, IPT_DIAL | IPF_PLAYER2, 100, 15, 0, 0, KEYCODE_N, KEYCODE_M, 0, 0 )

INPUT_PORTS_END

static READ16_HANDLER( pl2_port_0_word_r )
{
	if(readpaddle != 0)
		return readinputport(0);
	else
		return readinputport(3) + (readinputport(4) << 8);
}



static struct GfxLayout layout8x8 =
{
	8,8,
	RGN_FRAC(1,1),
	4,
	{ GFX_RAW },
	{ 4*8 },	/* org displacement - 8x8 tiles are taken from the RIGHT side of the 16x16 tile
				   (fixes cawing which uses character 0x0002 as space, typo instead of 0x20?) */
	{ 8*8 },	/* line modulo */
	64*8		/* char modulo */
};

static struct GfxLayout layout16x16 =
{
	16,16,
	RGN_FRAC(1,1),
	4,
	{ GFX_RAW },
	{ 0 },		/* org displacement */
	{ 8*8 },	/* line modulo */
	128*8		/* char modulo */
};

static struct GfxLayout layout32x32 =
{
	32,32,
	RGN_FRAC(1,1),
	4,
	{ GFX_RAW },
	{ 0 },		/* org displacement */
	{ 16*8 },	/* line modulo */
	512*8		/* char modulo */
};

static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &layout8x8,   0, 0x100 },
	{ REGION_GFX1, 0, &layout16x16, 0, 0x100 },
	{ REGION_GFX1, 0, &layout32x32, 0, 0x100 },
	{ -1 }
};



static MACHINE_DRIVER_START( cps2 )

	/* basic machine hardware */
	MDRV_CPU_ADD(M68000, 11800000)

	MDRV_CPU_MEMORY(cps2_readmem,cps2_writemem)
	MDRV_CPU_VBLANK_INT(cps2_interrupt,262)	// 262  /* ??? interrupts per frame */

	MDRV_CPU_ADD(Z80, 8000000)
	MDRV_CPU_MEMORY(qsound_readmem,qsound_writemem)
	MDRV_CPU_PERIODIC_INT(irq0_line_hold,251)	/* ?? */

	MDRV_FRAMES_PER_SECOND(59.633333)
	MDRV_VBLANK_DURATION(0)		//ks
//	MDRV_INTERLEAVE(262)  /* 262 scanlines, for raster effects */

	MDRV_NVRAM_HANDLER(cps2)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN | VIDEO_UPDATE_AFTER_VBLANK)		//ks
	MDRV_SCREEN_SIZE(64*8, 32*8)
	MDRV_VISIBLE_AREA(8*8, (64-8)*8-1, 2*8, 30*8-1 )
	MDRV_GFXDECODE(gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(4096)

	MDRV_VIDEO_START(cps2)
	MDRV_VIDEO_EOF(cps1)
	MDRV_VIDEO_UPDATE(cps1)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(QSOUND, qsound_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( dima )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(cps2)
	MDRV_CPU_REPLACE("main", M68000, 13000000)
MACHINE_DRIVER_END


ROM_START( 1944 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "nffu.03", 0x000000, 0x80000, 0x9693cf8f )
	ROM_LOAD16_WORD_SWAP( "nff.04",  0x080000, 0x80000, 0xdba1c66e )
	ROM_LOAD16_WORD_SWAP( "nffu.05", 0x100000, 0x80000, 0xea813eb7 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "nff.13m",   0x0000000, 0x400000, 0xc9fca741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.15m",   0x0000002, 0x400000, 0xf809d898 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.17m",   0x0000004, 0x400000, 0x15ba4507 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.19m",   0x0000006, 0x400000, 0x3dd41b8c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.14m",   0x1000000, 0x100000, 0x3fe3a54b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.16m",   0x1000002, 0x100000, 0x565cd231 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.18m",   0x1000004, 0x100000, 0x63ca5988 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.20m",   0x1000006, 0x100000, 0x21eb8f3b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "nff.01",   0x00000, 0x08000, 0xd2e44318 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "nff.11m",   0x000000, 0x400000, 0x243e4e05 )
	ROM_LOAD16_WORD_SWAP( "nff.12m",   0x400000, 0x400000, 0x4fcf1600 )
ROM_END

ROM_START( 1944j )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "nffj.03", 0x000000, 0x80000, 0x247521ef )
	ROM_LOAD16_WORD_SWAP( "nff.04",  0x080000, 0x80000, 0xdba1c66e )
	ROM_LOAD16_WORD_SWAP( "nffj.05", 0x100000, 0x80000, 0x7f20c2ef )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "nff.13m",   0x0000000, 0x400000, 0xc9fca741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.15m",   0x0000002, 0x400000, 0xf809d898 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.17m",   0x0000004, 0x400000, 0x15ba4507 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.19m",   0x0000006, 0x400000, 0x3dd41b8c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.14m",   0x1000000, 0x100000, 0x3fe3a54b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.16m",   0x1000002, 0x100000, 0x565cd231 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.18m",   0x1000004, 0x100000, 0x63ca5988 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "nff.20m",   0x1000006, 0x100000, 0x21eb8f3b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "nff.01",   0x00000, 0x08000, 0xd2e44318 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "nff.11m",   0x000000, 0x400000, 0x243e4e05 )
	ROM_LOAD16_WORD_SWAP( "nff.12m",   0x400000, 0x400000, 0x4fcf1600 )
ROM_END

ROM_START( 19xx )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "19xu.03", 0x000000, 0x80000, 0x05955268 )
	ROM_LOAD16_WORD_SWAP( "19xu.04", 0x080000, 0x80000, 0x3111ab7f )
	ROM_LOAD16_WORD_SWAP( "19xu.05", 0x100000, 0x80000, 0x38df4a63 )
	ROM_LOAD16_WORD_SWAP( "19xu.06", 0x180000, 0x80000, 0x5c7e60d3 )
	ROM_LOAD16_WORD_SWAP( "19x.07",  0x200000, 0x80000, 0x61c0296c )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "19x.13m",   0x0000000, 0x080000, 0x427aeb18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.15m",   0x0000002, 0x080000, 0x63bdbf54 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.17m",   0x0000004, 0x080000, 0x2dfe18b5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.19m",   0x0000006, 0x080000, 0xcbef9579 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.14m",   0x0800000, 0x200000, 0xe916967c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.16m",   0x0800002, 0x200000, 0x6e75f3db , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.18m",   0x0800004, 0x200000, 0x2213e798 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.20m",   0x0800006, 0x200000, 0xab9d5b96 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "19x.01",   0x00000, 0x08000, 0xef55195e )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "19x.11m",   0x000000, 0x200000, 0xd38beef3 )
	ROM_LOAD16_WORD_SWAP( "19x.12m",   0x200000, 0x200000, 0xd47c96e2 )
ROM_END

ROM_START( 19xxj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "19xj.03a", 0x000000, 0x80000, 0xed08bdd1 )
	ROM_LOAD16_WORD_SWAP( "19xj.04a", 0x080000, 0x80000, 0xfb8e3f29 )
	ROM_LOAD16_WORD_SWAP( "19xj.05a", 0x100000, 0x80000, 0xaa508ac4 )
	ROM_LOAD16_WORD_SWAP( "19xj.06a", 0x180000, 0x80000, 0xff2d785b )
	ROM_LOAD16_WORD_SWAP( "19x.07",   0x200000, 0x80000, 0x61c0296c )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "19x.13m",   0x0000000, 0x080000, 0x427aeb18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.15m",   0x0000002, 0x080000, 0x63bdbf54 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.17m",   0x0000004, 0x080000, 0x2dfe18b5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.19m",   0x0000006, 0x080000, 0xcbef9579 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.14m",   0x0800000, 0x200000, 0xe916967c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.16m",   0x0800002, 0x200000, 0x6e75f3db , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.18m",   0x0800004, 0x200000, 0x2213e798 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.20m",   0x0800006, 0x200000, 0xab9d5b96 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "19x.01",   0x00000, 0x08000, 0xef55195e )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "19x.11m",   0x000000, 0x200000, 0xd38beef3 )
	ROM_LOAD16_WORD_SWAP( "19x.12m",   0x200000, 0x200000, 0xd47c96e2 )
ROM_END

ROM_START( 19xxjr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "19xj.03", 0x000000, 0x80000, 0x26a381ed )
	ROM_LOAD16_WORD_SWAP( "19xj.04", 0x080000, 0x80000, 0x30100cca )
	ROM_LOAD16_WORD_SWAP( "19xj.05", 0x100000, 0x80000, 0xde67e938 )
	ROM_LOAD16_WORD_SWAP( "19xj.06", 0x180000, 0x80000, 0x39f9a409 )
	ROM_LOAD16_WORD_SWAP( "19x.07",  0x200000, 0x80000, 0x61c0296c )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "19x.13m",   0x0000000, 0x080000, 0x427aeb18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.15m",   0x0000002, 0x080000, 0x63bdbf54 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.17m",   0x0000004, 0x080000, 0x2dfe18b5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.19m",   0x0000006, 0x080000, 0xcbef9579 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.14m",   0x0800000, 0x200000, 0xe916967c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.16m",   0x0800002, 0x200000, 0x6e75f3db , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.18m",   0x0800004, 0x200000, 0x2213e798 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.20m",   0x0800006, 0x200000, 0xab9d5b96 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "19x.01",   0x00000, 0x08000, 0xef55195e )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "19x.11m",   0x000000, 0x200000, 0xd38beef3 )
	ROM_LOAD16_WORD_SWAP( "19x.12m",   0x200000, 0x200000, 0xd47c96e2 )
ROM_END

ROM_START( 19xxa )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "19xa.03", 0x000000, 0x80000, 0x0c20fd50 )
	ROM_LOAD16_WORD_SWAP( "19xa.04", 0x080000, 0x80000, 0x1fc37508 )
	ROM_LOAD16_WORD_SWAP( "19xa.05", 0x100000, 0x80000, 0x6c9cc4ed )
	ROM_LOAD16_WORD_SWAP( "19xa.06", 0x180000, 0x80000, 0xca5b9f76 )
	ROM_LOAD16_WORD_SWAP( "19x.07",  0x200000, 0x80000, 0x61c0296c )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "19x.13m",   0x0000000, 0x080000, 0x427aeb18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.15m",   0x0000002, 0x080000, 0x63bdbf54 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.17m",   0x0000004, 0x080000, 0x2dfe18b5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.19m",   0x0000006, 0x080000, 0xcbef9579 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.14m",   0x0800000, 0x200000, 0xe916967c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.16m",   0x0800002, 0x200000, 0x6e75f3db , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.18m",   0x0800004, 0x200000, 0x2213e798 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.20m",   0x0800006, 0x200000, 0xab9d5b96 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "19x.01",   0x00000, 0x08000, 0xef55195e )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "19x.11m",   0x000000, 0x200000, 0xd38beef3 )
	ROM_LOAD16_WORD_SWAP( "19x.12m",   0x200000, 0x200000, 0xd47c96e2 )
ROM_END

ROM_START( 19xxh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "19xh.03a", 0x000000, 0x80000, 0x357be2ac )
	ROM_LOAD16_WORD_SWAP( "19xh.04a", 0x080000, 0x80000, 0xbb13ea3b )
	ROM_LOAD16_WORD_SWAP( "19xh.05a", 0x100000, 0x80000, 0xcbd76601 )
	ROM_LOAD16_WORD_SWAP( "19xh.06a", 0x180000, 0x80000, 0xb362de8b )
	ROM_LOAD16_WORD_SWAP( "19x.07",   0x200000, 0x80000, 0x61c0296c )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "19x.13m",   0x0000000, 0x080000, 0x427aeb18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.15m",   0x0000002, 0x080000, 0x63bdbf54 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.17m",   0x0000004, 0x080000, 0x2dfe18b5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.19m",   0x0000006, 0x080000, 0xcbef9579 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.14m",   0x0800000, 0x200000, 0xe916967c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.16m",   0x0800002, 0x200000, 0x6e75f3db , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.18m",   0x0800004, 0x200000, 0x2213e798 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.20m",   0x0800006, 0x200000, 0xab9d5b96 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "19x.01",   0x00000, 0x08000, 0xef55195e )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "19x.11m",   0x000000, 0x200000, 0xd38beef3 )
	ROM_LOAD16_WORD_SWAP( "19x.12m",   0x200000, 0x200000, 0xd47c96e2 )
ROM_END

ROM_START( 19xxb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "19xb.03a", 0x000000, 0x80000, 0x341bdf4a )
	ROM_LOAD16_WORD_SWAP( "19xb.04a", 0x080000, 0x80000, 0xdff8069e )
	ROM_LOAD16_WORD_SWAP( "19xb.05a", 0x100000, 0x80000, 0xa47a92a8 )
	ROM_LOAD16_WORD_SWAP( "19xb.06a", 0x180000, 0x80000, 0xc52df10d )
	ROM_LOAD16_WORD_SWAP( "19x.07",   0x200000, 0x80000, 0x61c0296c )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "19x.13m",   0x0000000, 0x080000, 0x427aeb18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.15m",   0x0000002, 0x080000, 0x63bdbf54 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.17m",   0x0000004, 0x080000, 0x2dfe18b5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.19m",   0x0000006, 0x080000, 0xcbef9579 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.14m",   0x0800000, 0x200000, 0xe916967c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.16m",   0x0800002, 0x200000, 0x6e75f3db , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.18m",   0x0800004, 0x200000, 0x2213e798 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "19x.20m",   0x0800006, 0x200000, 0xab9d5b96 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "19x.01",   0x00000, 0x08000, 0xef55195e )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "19x.11m",   0x000000, 0x200000, 0xd38beef3 )
	ROM_LOAD16_WORD_SWAP( "19x.12m",   0x200000, 0x200000, 0xd47c96e2 )
ROM_END

ROM_START( armwar )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pwge.03c", 0x000000, 0x80000, 0x31f74931 )
	ROM_LOAD16_WORD_SWAP( "pwge.04c", 0x080000, 0x80000, 0x16f34f5f )
	ROM_LOAD16_WORD_SWAP( "pwge.05b", 0x100000, 0x80000, 0x4403ed08 )
	ROM_LOAD16_WORD_SWAP( "pwg.06",   0x180000, 0x80000, 0x87a60ce8 )
	ROM_LOAD16_WORD_SWAP( "pwg.07",   0x200000, 0x80000, 0xf7b148df )
	ROM_LOAD16_WORD_SWAP( "pwg.08",   0x280000, 0x80000, 0xcc62823e )
	ROM_LOAD16_WORD_SWAP( "pwg.09a",  0x300000, 0x80000, 0x4c26baee )
	ROM_LOAD16_WORD_SWAP( "pwg.10",   0x380000, 0x80000, 0x07c4fb28 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pwg.13m",   0x0000000, 0x400000, 0xae8fe08e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.15m",   0x0000002, 0x400000, 0xdb560f58 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.17m",   0x0000004, 0x400000, 0xbc475b94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.19m",   0x0000006, 0x400000, 0x07439ff7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.14m",   0x1000000, 0x100000, 0xc3f9ba63 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.16m",   0x1000002, 0x100000, 0x815b0e7b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.18m",   0x1000004, 0x100000, 0x0109c71b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.20m",   0x1000006, 0x100000, 0xeb75ffbe , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pwg.01",   0x00000, 0x08000, 0x18a5c0e4 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pwg.02",   0x28000, 0x20000, 0xc9dfffa6 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pwg.11m",   0x000000, 0x200000, 0xa78f7433 )
	ROM_LOAD16_WORD_SWAP( "pwg.12m",   0x200000, 0x200000, 0x77438ed0 )
ROM_END

ROM_START( armwarr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pwge.03b", 0x000000, 0x80000, 0xe822e3e9 )
	ROM_LOAD16_WORD_SWAP( "pwge.04b", 0x080000, 0x80000, 0x4f89de39 )
	ROM_LOAD16_WORD_SWAP( "pwge.05a", 0x100000, 0x80000, 0x83df24e5 )
	ROM_LOAD16_WORD_SWAP( "pwg.06",   0x180000, 0x80000, 0x87a60ce8 )
	ROM_LOAD16_WORD_SWAP( "pwg.07",   0x200000, 0x80000, 0xf7b148df )
	ROM_LOAD16_WORD_SWAP( "pwg.08",   0x280000, 0x80000, 0xcc62823e )
	ROM_LOAD16_WORD_SWAP( "pwg.09",   0x300000, 0x80000, 0xddc85ca6 )
	ROM_LOAD16_WORD_SWAP( "pwg.10",   0x380000, 0x80000, 0x07c4fb28 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pwg.13m",   0x0000000, 0x400000, 0xae8fe08e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.15m",   0x0000002, 0x400000, 0xdb560f58 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.17m",   0x0000004, 0x400000, 0xbc475b94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.19m",   0x0000006, 0x400000, 0x07439ff7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.14m",   0x1000000, 0x100000, 0xc3f9ba63 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.16m",   0x1000002, 0x100000, 0x815b0e7b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.18m",   0x1000004, 0x100000, 0x0109c71b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.20m",   0x1000006, 0x100000, 0xeb75ffbe , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pwg.01",   0x00000, 0x08000, 0x18a5c0e4 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pwg.02",   0x28000, 0x20000, 0xc9dfffa6 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pwg.11m",   0x000000, 0x200000, 0xa78f7433 )
	ROM_LOAD16_WORD_SWAP( "pwg.12m",   0x200000, 0x200000, 0x77438ed0 )
ROM_END

ROM_START( armwaru )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pwgu.03b", 0x000000, 0x80000, 0x8b95497a )
	ROM_LOAD16_WORD_SWAP( "pwgu.04b", 0x080000, 0x80000, 0x29eb5661 )
	ROM_LOAD16_WORD_SWAP( "pwgu.05b", 0x100000, 0x80000, 0xa54e9e44 )
	ROM_LOAD16_WORD_SWAP( "pwg.06",   0x180000, 0x80000, 0x87a60ce8 )
	ROM_LOAD16_WORD_SWAP( "pwg.07",   0x200000, 0x80000, 0xf7b148df )
	ROM_LOAD16_WORD_SWAP( "pwg.08",   0x280000, 0x80000, 0xcc62823e )
	ROM_LOAD16_WORD_SWAP( "pwg.09a",  0x300000, 0x80000, 0x4c26baee )
	ROM_LOAD16_WORD_SWAP( "pwg.10",   0x380000, 0x80000, 0x07c4fb28 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pwg.13m",   0x0000000, 0x400000, 0xae8fe08e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.15m",   0x0000002, 0x400000, 0xdb560f58 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.17m",   0x0000004, 0x400000, 0xbc475b94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.19m",   0x0000006, 0x400000, 0x07439ff7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.14m",   0x1000000, 0x100000, 0xc3f9ba63 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.16m",   0x1000002, 0x100000, 0x815b0e7b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.18m",   0x1000004, 0x100000, 0x0109c71b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.20m",   0x1000006, 0x100000, 0xeb75ffbe , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pwg.01",   0x00000, 0x08000, 0x18a5c0e4 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pwg.02",   0x28000, 0x20000, 0xc9dfffa6 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pwg.11m",   0x000000, 0x200000, 0xa78f7433 )
	ROM_LOAD16_WORD_SWAP( "pwg.12m",   0x200000, 0x200000, 0x77438ed0 )
ROM_END

ROM_START( pgear )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pwgj.03a", 0x000000, 0x80000, 0xc79c0c02 )
	ROM_LOAD16_WORD_SWAP( "pwgj.04a", 0x080000, 0x80000, 0x167c6ed8 )
	ROM_LOAD16_WORD_SWAP( "pwgj.05a", 0x100000, 0x80000, 0xa63fb400 )
	ROM_LOAD16_WORD_SWAP( "pwg.06",   0x180000, 0x80000, 0x87a60ce8 )
	ROM_LOAD16_WORD_SWAP( "pwg.07",   0x200000, 0x80000, 0xf7b148df )
	ROM_LOAD16_WORD_SWAP( "pwg.08",   0x280000, 0x80000, 0xcc62823e )
	ROM_LOAD16_WORD_SWAP( "pwg.09a",  0x300000, 0x80000, 0x4c26baee )
	ROM_LOAD16_WORD_SWAP( "pwg.10",   0x380000, 0x80000, 0x07c4fb28 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pwg.13m",   0x0000000, 0x400000, 0xae8fe08e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.15m",   0x0000002, 0x400000, 0xdb560f58 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.17m",   0x0000004, 0x400000, 0xbc475b94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.19m",   0x0000006, 0x400000, 0x07439ff7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.14m",   0x1000000, 0x100000, 0xc3f9ba63 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.16m",   0x1000002, 0x100000, 0x815b0e7b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.18m",   0x1000004, 0x100000, 0x0109c71b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.20m",   0x1000006, 0x100000, 0xeb75ffbe , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pwg.01",   0x00000, 0x08000, 0x18a5c0e4 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pwg.02",   0x28000, 0x20000, 0xc9dfffa6 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pwg.11m",   0x000000, 0x200000, 0xa78f7433 )
	ROM_LOAD16_WORD_SWAP( "pwg.12m",   0x200000, 0x200000, 0x77438ed0 )
ROM_END

ROM_START( pgearr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pwgj.03", 0x000000, 0x80000, 0xf264e74b )
	ROM_LOAD16_WORD_SWAP( "pwgj.04", 0x080000, 0x80000, 0x23a84983 )
	ROM_LOAD16_WORD_SWAP( "pwgj.05", 0x100000, 0x80000, 0xbef58c62 )
	ROM_LOAD16_WORD_SWAP( "pwg.06",  0x180000, 0x80000, 0x87a60ce8 )
	ROM_LOAD16_WORD_SWAP( "pwg.07",  0x200000, 0x80000, 0xf7b148df )
	ROM_LOAD16_WORD_SWAP( "pwg.08",  0x280000, 0x80000, 0xcc62823e )
	ROM_LOAD16_WORD_SWAP( "pwg.09",  0x300000, 0x80000, 0xddc85ca6 )
	ROM_LOAD16_WORD_SWAP( "pwg.10",  0x380000, 0x80000, 0x07c4fb28 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pwg.13m",   0x0000000, 0x400000, 0xae8fe08e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.15m",   0x0000002, 0x400000, 0xdb560f58 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.17m",   0x0000004, 0x400000, 0xbc475b94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.19m",   0x0000006, 0x400000, 0x07439ff7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.14m",   0x1000000, 0x100000, 0xc3f9ba63 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.16m",   0x1000002, 0x100000, 0x815b0e7b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.18m",   0x1000004, 0x100000, 0x0109c71b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.20m",   0x1000006, 0x100000, 0xeb75ffbe , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pwg.01",   0x00000, 0x08000, 0x18a5c0e4 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pwg.02",   0x28000, 0x20000, 0xc9dfffa6 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pwg.11m",   0x000000, 0x200000, 0xa78f7433 )
	ROM_LOAD16_WORD_SWAP( "pwg.12m",   0x200000, 0x200000, 0x77438ed0 )
ROM_END

ROM_START( armwara )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pwga.03a", 0x000000, 0x80000, 0x8d474ab1 )
	ROM_LOAD16_WORD_SWAP( "pwga.04a", 0x080000, 0x80000, 0x81b5aec7 )
	ROM_LOAD16_WORD_SWAP( "pwga.05a", 0x100000, 0x80000, 0x2618e819 )
	ROM_LOAD16_WORD_SWAP( "pwg.06",   0x180000, 0x80000, 0x87a60ce8 )
	ROM_LOAD16_WORD_SWAP( "pwg.07",   0x200000, 0x80000, 0xf7b148df )
	ROM_LOAD16_WORD_SWAP( "pwg.08",   0x280000, 0x80000, 0xcc62823e )
	ROM_LOAD16_WORD_SWAP( "pwg.09",   0x300000, 0x80000, 0xddc85ca6 )
	ROM_LOAD16_WORD_SWAP( "pwg.10",   0x380000, 0x80000, 0x07c4fb28 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pwg.13m",   0x0000000, 0x400000, 0xae8fe08e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.15m",   0x0000002, 0x400000, 0xdb560f58 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.17m",   0x0000004, 0x400000, 0xbc475b94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.19m",   0x0000006, 0x400000, 0x07439ff7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.14m",   0x1000000, 0x100000, 0xc3f9ba63 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.16m",   0x1000002, 0x100000, 0x815b0e7b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.18m",   0x1000004, 0x100000, 0x0109c71b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pwg.20m",   0x1000006, 0x100000, 0xeb75ffbe , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pwg.01",   0x00000, 0x08000, 0x18a5c0e4 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pwg.02",   0x28000, 0x20000, 0xc9dfffa6 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pwg.11m",   0x000000, 0x200000, 0xa78f7433 )
	ROM_LOAD16_WORD_SWAP( "pwg.12m",   0x200000, 0x200000, 0x77438ed0 )
ROM_END

ROM_START( avsp )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "avpe.03d", 0x000000, 0x80000, 0x774334a9 )
	ROM_LOAD16_WORD_SWAP( "avpe.04d", 0x080000, 0x80000, 0x7fa83769 )
	ROM_LOAD16_WORD_SWAP( "avp.05d",  0x100000, 0x80000, 0xfbfb5d7a )
	ROM_LOAD16_WORD_SWAP( "avp.06",   0x180000, 0x80000, 0x190b817f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "avp.13m",   0x0000000, 0x200000, 0x8f8b5ae4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.15m",   0x0000002, 0x200000, 0xb00280df , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.17m",   0x0000004, 0x200000, 0x94403195 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.19m",   0x0000006, 0x200000, 0xe1981245 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.14m",   0x0800000, 0x200000, 0xebba093e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.16m",   0x0800002, 0x200000, 0xfb228297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.18m",   0x0800004, 0x200000, 0x34fb7232 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.20m",   0x0800006, 0x200000, 0xf90baa21 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "avp.01",   0x00000, 0x08000, 0x2d3b4220 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "avp.11m",   0x000000, 0x200000, 0x83499817 )
	ROM_LOAD16_WORD_SWAP( "avp.12m",   0x200000, 0x200000, 0xf4110d49 )
ROM_END

ROM_START( avspu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "avpu.03d", 0x000000, 0x80000, 0x42757950 )
	ROM_LOAD16_WORD_SWAP( "avpu.04d", 0x080000, 0x80000, 0x5abcdee6 )
	ROM_LOAD16_WORD_SWAP( "avp.05d",  0x100000, 0x80000, 0xfbfb5d7a )
	ROM_LOAD16_WORD_SWAP( "avp.06",   0x180000, 0x80000, 0x190b817f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "avp.13m",   0x0000000, 0x200000, 0x8f8b5ae4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.15m",   0x0000002, 0x200000, 0xb00280df , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.17m",   0x0000004, 0x200000, 0x94403195 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.19m",   0x0000006, 0x200000, 0xe1981245 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.14m",   0x0800000, 0x200000, 0xebba093e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.16m",   0x0800002, 0x200000, 0xfb228297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.18m",   0x0800004, 0x200000, 0x34fb7232 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.20m",   0x0800006, 0x200000, 0xf90baa21 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "avp.01",   0x00000, 0x08000, 0x2d3b4220 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "avp.11m",   0x000000, 0x200000, 0x83499817 )
	ROM_LOAD16_WORD_SWAP( "avp.12m",   0x200000, 0x200000, 0xf4110d49 )
ROM_END

ROM_START( avspj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "avpj.03d", 0x000000, 0x80000, 0x49799119 )
	ROM_LOAD16_WORD_SWAP( "avpj.04d", 0x080000, 0x80000, 0x8cd2bba8 )
	ROM_LOAD16_WORD_SWAP( "avp.05d",  0x100000, 0x80000, 0xfbfb5d7a )
	ROM_LOAD16_WORD_SWAP( "avp.06",   0x180000, 0x80000, 0x190b817f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "avp.13m",   0x0000000, 0x200000, 0x8f8b5ae4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.15m",   0x0000002, 0x200000, 0xb00280df , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.17m",   0x0000004, 0x200000, 0x94403195 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.19m",   0x0000006, 0x200000, 0xe1981245 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.14m",   0x0800000, 0x200000, 0xebba093e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.16m",   0x0800002, 0x200000, 0xfb228297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.18m",   0x0800004, 0x200000, 0x34fb7232 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.20m",   0x0800006, 0x200000, 0xf90baa21 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "avp.01",   0x00000, 0x08000, 0x2d3b4220 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "avp.11m",   0x000000, 0x200000, 0x83499817 )
	ROM_LOAD16_WORD_SWAP( "avp.12m",   0x200000, 0x200000, 0xf4110d49 )
ROM_END

ROM_START( avspa )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "avpa.03d", 0x000000, 0x80000, 0x6c1c1858 )
	ROM_LOAD16_WORD_SWAP( "avpa.04d", 0x080000, 0x80000, 0x94f50b0c )
	ROM_LOAD16_WORD_SWAP( "avp.05d",  0x100000, 0x80000, 0xfbfb5d7a )
	ROM_LOAD16_WORD_SWAP( "avp.06",   0x180000, 0x80000, 0x190b817f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "avp.13m",   0x0000000, 0x200000, 0x8f8b5ae4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.15m",   0x0000002, 0x200000, 0xb00280df , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.17m",   0x0000004, 0x200000, 0x94403195 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.19m",   0x0000006, 0x200000, 0xe1981245 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.14m",   0x0800000, 0x200000, 0xebba093e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.16m",   0x0800002, 0x200000, 0xfb228297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.18m",   0x0800004, 0x200000, 0x34fb7232 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.20m",   0x0800006, 0x200000, 0xf90baa21 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "avp.01",   0x00000, 0x08000, 0x2d3b4220 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "avp.11m",   0x000000, 0x200000, 0x83499817 )
	ROM_LOAD16_WORD_SWAP( "avp.12m",   0x200000, 0x200000, 0xf4110d49 )
ROM_END

ROM_START( avsph )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "avph.03d", 0x000000, 0x80000, 0x3e440447 )
	ROM_LOAD16_WORD_SWAP( "avph.04d", 0x080000, 0x80000, 0xaf6fc82f )
	ROM_LOAD16_WORD_SWAP( "avp.05d",  0x100000, 0x80000, 0xfbfb5d7a )
	ROM_LOAD16_WORD_SWAP( "avp.06",   0x180000, 0x80000, 0x190b817f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "avp.13m",   0x0000000, 0x200000, 0x8f8b5ae4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.15m",   0x0000002, 0x200000, 0xb00280df , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.17m",   0x0000004, 0x200000, 0x94403195 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.19m",   0x0000006, 0x200000, 0xe1981245 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.14m",   0x0800000, 0x200000, 0xebba093e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.16m",   0x0800002, 0x200000, 0xfb228297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.18m",   0x0800004, 0x200000, 0x34fb7232 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "avp.20m",   0x0800006, 0x200000, 0xf90baa21 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "avp.01",   0x00000, 0x08000, 0x2d3b4220 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "avp.11m",   0x000000, 0x200000, 0x83499817 )
	ROM_LOAD16_WORD_SWAP( "avp.12m",   0x200000, 0x200000, 0xf4110d49 )
ROM_END

ROM_START( batcir )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "btce.03", 0x000000, 0x80000, 0xbc60484b )
	ROM_LOAD16_WORD_SWAP( "btce.04", 0x080000, 0x80000, 0x457d55f6 )
	ROM_LOAD16_WORD_SWAP( "btce.05", 0x100000, 0x80000, 0xe86560d7 )
	ROM_LOAD16_WORD_SWAP( "btce.06", 0x180000, 0x80000, 0xf778e61b )
	ROM_LOAD16_WORD_SWAP( "btc.07",  0x200000, 0x80000, 0x7322d5db )
	ROM_LOAD16_WORD_SWAP( "btc.08",  0x280000, 0x80000, 0x6aac85ab )
	ROM_LOAD16_WORD_SWAP( "btc.09",  0x300000, 0x80000, 0x1203db08 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "btc.13m",   0x000000, 0x400000, 0xdc705bad , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "btc.15m",   0x000002, 0x400000, 0xe5779a3c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "btc.17m",   0x000004, 0x400000, 0xb33f4112 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "btc.19m",   0x000006, 0x400000, 0xa6fcdb7e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "btc.01",   0x00000, 0x08000, 0x1e194310 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "btc.02",   0x28000, 0x20000, 0x01aeb8e6 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "btc.11m",   0x000000, 0x200000, 0xc27f2229 )
	ROM_LOAD16_WORD_SWAP( "btc.12m",   0x200000, 0x200000, 0x418a2e33 )
ROM_END

ROM_START( batcirj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "btcj.03", 0x000000, 0x80000, 0x6b7e168d )
	ROM_LOAD16_WORD_SWAP( "btcj.04", 0x080000, 0x80000, 0x46ba3467 )
	ROM_LOAD16_WORD_SWAP( "btcj.05", 0x100000, 0x80000, 0x0e23a859 )
	ROM_LOAD16_WORD_SWAP( "btcj.06", 0x180000, 0x80000, 0xa853b59c )
	ROM_LOAD16_WORD_SWAP( "btc.07",  0x200000, 0x80000, 0x7322d5db )
	ROM_LOAD16_WORD_SWAP( "btc.08",  0x280000, 0x80000, 0x6aac85ab )
	ROM_LOAD16_WORD_SWAP( "btc.09",  0x300000, 0x80000, 0x1203db08 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "btc.13m",   0x000000, 0x400000, 0xdc705bad , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "btc.15m",   0x000002, 0x400000, 0xe5779a3c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "btc.17m",   0x000004, 0x400000, 0xb33f4112 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "btc.19m",   0x000006, 0x400000, 0xa6fcdb7e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "btc.01",   0x00000, 0x08000, 0x1e194310 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "btc.02",   0x28000, 0x20000, 0x01aeb8e6 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "btc.11m",   0x000000, 0x200000, 0xc27f2229 )
	ROM_LOAD16_WORD_SWAP( "btc.12m",   0x200000, 0x200000, 0x418a2e33 )
ROM_END

ROM_START( batcira )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "btca.03", 0x000000, 0x80000, 0x1ad20d87 )
	ROM_LOAD16_WORD_SWAP( "btca.04", 0x080000, 0x80000, 0x2b3f4dbe )
	ROM_LOAD16_WORD_SWAP( "btca.05", 0x100000, 0x80000, 0x8238a3d9 )
	ROM_LOAD16_WORD_SWAP( "btca.06", 0x180000, 0x80000, 0x446c7c02 )
	ROM_LOAD16_WORD_SWAP( "btc.07",  0x200000, 0x80000, 0x7322d5db )
	ROM_LOAD16_WORD_SWAP( "btc.08",  0x280000, 0x80000, 0x6aac85ab )
	ROM_LOAD16_WORD_SWAP( "btc.09",  0x300000, 0x80000, 0x1203db08 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "btc.13m",   0x000000, 0x400000, 0xdc705bad , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "btc.15m",   0x000002, 0x400000, 0xe5779a3c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "btc.17m",   0x000004, 0x400000, 0xb33f4112 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "btc.19m",   0x000006, 0x400000, 0xa6fcdb7e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "btc.01",   0x00000, 0x08000, 0x1e194310 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "btc.02",   0x28000, 0x20000, 0x01aeb8e6 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "btc.11m",   0x000000, 0x200000, 0xc27f2229 )
	ROM_LOAD16_WORD_SWAP( "btc.12m",   0x200000, 0x200000, 0x418a2e33 )
ROM_END

ROM_START( choko )
	ROM_REGION( CODE_SIZE, REGION_CPU1, ROMREGION_ERASEFF )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "tkoj.03", 0x000000, 0x80000, 0x11f5452f )
	ROM_LOAD16_WORD_SWAP( "tkoj.04", 0x080000, 0x80000, 0x68655378 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "tkoj1_d.simm1",   0x0000000, 0x200000, 0x6933377d , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "tkoj1_c.simm1",   0x0000001, 0x200000, 0x7f668950 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "tkoj1_b.simm1",   0x0000002, 0x200000, 0xcfb68ca9 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "tkoj1_a.simm1",   0x0000003, 0x200000, 0x437e21c5 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "tkoj3_d.simm3",   0x0000004, 0x200000, 0xa9e32b57 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "tkoj3_c.simm3",   0x0000005, 0x200000, 0xb7ab9338 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "tkoj3_b.simm3",   0x0000006, 0x200000, 0x4d3f919a , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "tkoj3_a.simm3",   0x0000007, 0x200000, 0xcfef17ab , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "tko.01",   0x00000, 0x08000, 0x6eda50c2 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_BYTE( "tkoj5_a.simm5",   0x000000, 0x200000, 0xab45d509 ) // ROM on a simm
	ROM_LOAD16_BYTE( "tkoj5_b.simm5",   0x000001, 0x200000, 0xfa905c3d ) // ROM on a simm
	/* Same Content, different interleave */
//  ROM_LOAD16_WORD_SWAP( "tko-simm.05a",   0x000000, 0x200000, 0xa275a387 ) // ROM on a simm
//  ROM_LOAD16_WORD_SWAP( "tko-simm.05b",   0x200000, 0x200000, 0xcb686c23 ) // ROM on a simm
ROM_END

ROM_START( csclubj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "cscj.03", 0x000000, 0x80000, 0xec4ddaa2 )
	ROM_LOAD16_WORD_SWAP( "cscj.04", 0x080000, 0x80000, 0x60c632bb )
	ROM_LOAD16_WORD_SWAP( "cscj.05", 0x100000, 0x80000, 0xad042003 )
	ROM_LOAD16_WORD_SWAP( "cscj.06", 0x180000, 0x80000, 0x169e4d40 )
	ROM_LOAD16_WORD_SWAP( "csc.07",  0x200000, 0x80000, 0x01b05caa )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "csc.14m",   0x800000, 0x200000, 0xe8904afa , ROM_GROUPWORD | ROM_SKIP(6) ) /* roms 73 to 76 joined in all eprom version */
	ROMX_LOAD( "csc.16m",   0x800002, 0x200000, 0xc98c8079 , ROM_GROUPWORD | ROM_SKIP(6) ) /* roms 63 to 66 joined in all eprom version */
	ROMX_LOAD( "csc.18m",   0x800004, 0x200000, 0xc030df5a , ROM_GROUPWORD | ROM_SKIP(6) ) /* roms 83 to 86 joined in all eprom version */
	ROMX_LOAD( "csc.20m",   0x800006, 0x200000, 0xb4e55863 , ROM_GROUPWORD | ROM_SKIP(6) ) /* roms 93 to 96 joined in all eprom version */

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "csc.01",   0x00000, 0x08000, 0xee162111 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "csc.11m",   0x000000, 0x200000, 0xa027b827 ) /* roms 51 to 54 joined in all eprom version */
	ROM_LOAD16_WORD_SWAP( "csc.12m",   0x200000, 0x200000, 0xcb7f6e55 ) /* roms 55 to 58 joined in all eprom version */
ROM_END

ROM_START( csclub )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "csce.03", 0x000000, 0x80000, 0xf2c852ef )
	ROM_LOAD16_WORD_SWAP( "csce.04", 0x080000, 0x80000, 0x1184530f )
	ROM_LOAD16_WORD_SWAP( "csce.05", 0x100000, 0x80000, 0x804e2b6b )
	ROM_LOAD16_WORD_SWAP( "csce.06", 0x180000, 0x80000, 0x09277cb9 )
	ROM_LOAD16_WORD_SWAP( "csc.07",  0x200000, 0x80000, 0x01b05caa )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "csc.14m",   0x800000, 0x200000, 0xe8904afa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "csc.16m",   0x800002, 0x200000, 0xc98c8079 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "csc.18m",   0x800004, 0x200000, 0xc030df5a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "csc.20m",   0x800006, 0x200000, 0xb4e55863 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "csc.01",   0x00000, 0x08000, 0xee162111 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "csc.11m",   0x000000, 0x200000, 0xa027b827 )
	ROM_LOAD16_WORD_SWAP( "csc.12m",   0x200000, 0x200000, 0xcb7f6e55 )
ROM_END

ROM_START( cscluba )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "csca.03", 0x000000, 0x80000, 0xb6acd708 )
	ROM_LOAD16_WORD_SWAP( "csca.04", 0x080000, 0x80000, 0xd44ae35f )
	ROM_LOAD16_WORD_SWAP( "csca.05", 0x100000, 0x80000, 0x8da76aec )
	ROM_LOAD16_WORD_SWAP( "csca.06", 0x180000, 0x80000, 0xa1b7b1ee )
	ROM_LOAD16_WORD_SWAP( "csc.07",  0x200000, 0x80000, 0x01b05caa )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "csc.14m",   0x800000, 0x200000, 0xe8904afa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "csc.16m",   0x800002, 0x200000, 0xc98c8079 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "csc.18m",   0x800004, 0x200000, 0xc030df5a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "csc.20m",   0x800006, 0x200000, 0xb4e55863 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "csc.01",   0x00000, 0x08000, 0xee162111 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "csc.11m",   0x000000, 0x200000, 0xa027b827 )
	ROM_LOAD16_WORD_SWAP( "csc.12m",   0x200000, 0x200000, 0xcb7f6e55 )
ROM_END

ROM_START( csclubh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "csch.03", 0x000000, 0x80000, 0x0dd7e46d )
	ROM_LOAD16_WORD_SWAP( "csch.04", 0x080000, 0x80000, 0x486e8143 )
	ROM_LOAD16_WORD_SWAP( "csch.05", 0x100000, 0x80000, 0x9e509dfb )
	ROM_LOAD16_WORD_SWAP( "csch.06", 0x180000, 0x80000, 0x817ba313 )
	ROM_LOAD16_WORD_SWAP( "csc.07",  0x200000, 0x80000, 0x01b05caa )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "csc.14m",   0x800000, 0x200000, 0xe8904afa , ROM_GROUPWORD | ROM_SKIP(6) ) /* roms 73 to 76 joined in all eprom version */
	ROMX_LOAD( "csc.16m",   0x800002, 0x200000, 0xc98c8079 , ROM_GROUPWORD | ROM_SKIP(6) ) /* roms 63 to 66 joined in all eprom version */
	ROMX_LOAD( "csc.18m",   0x800004, 0x200000, 0xc030df5a , ROM_GROUPWORD | ROM_SKIP(6) ) /* roms 83 to 86 joined in all eprom version */
	ROMX_LOAD( "csc.20m",   0x800006, 0x200000, 0xb4e55863 , ROM_GROUPWORD | ROM_SKIP(6) ) /* roms 93 to 96 joined in all eprom version */

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "csc.01",   0x00000, 0x08000, 0xee162111 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "csc.11m",   0x000000, 0x200000, 0xa027b827 ) /* roms 51 to 54 joined in all eprom version */
	ROM_LOAD16_WORD_SWAP( "csc.12m",   0x200000, 0x200000, 0xcb7f6e55 ) /* roms 55 to 58 joined in all eprom version */
ROM_END

ROM_START( cybots )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "cybe.03", 0x000000, 0x80000, 0x234381cd )
	ROM_LOAD16_WORD_SWAP( "cybe.04", 0x080000, 0x80000, 0x80691061 )
	ROM_LOAD16_WORD_SWAP( "cyb.05",  0x100000, 0x80000, 0xec40408e )
	ROM_LOAD16_WORD_SWAP( "cyb.06",  0x180000, 0x80000, 0x1ad0bed2 )
	ROM_LOAD16_WORD_SWAP( "cyb.07",  0x200000, 0x80000, 0x6245a39a )
	ROM_LOAD16_WORD_SWAP( "cyb.08",  0x280000, 0x80000, 0x4b48e223 )
	ROM_LOAD16_WORD_SWAP( "cyb.09",  0x300000, 0x80000, 0xe15238f6 )
	ROM_LOAD16_WORD_SWAP( "cyb.10",  0x380000, 0x80000, 0x75f4003b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "cyb.13m",   0x0000000, 0x400000, 0xf0dce192 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.15m",   0x0000002, 0x400000, 0x187aa39c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.17m",   0x0000004, 0x400000, 0x8a0e4b12 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.19m",   0x0000006, 0x400000, 0x34b62612 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.14m",   0x1000000, 0x400000, 0xc1537957 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.16m",   0x1000002, 0x400000, 0x15349e86 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.18m",   0x1000004, 0x400000, 0xd83e977d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.20m",   0x1000006, 0x400000, 0x77cdad5c , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "cyb.01",   0x00000, 0x08000, 0x9c0fb079 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "cyb.02",   0x28000, 0x20000, 0x51cb0c4e )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "cyb.11m",   0x000000, 0x200000, 0x362ccab2 )
	ROM_LOAD16_WORD_SWAP( "cyb.12m",   0x200000, 0x200000, 0x7066e9cc )
ROM_END

ROM_START( cybotsu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "cybu.03", 0x000000, 0x80000, 0xdb4da8f4 )
	ROM_LOAD16_WORD_SWAP( "cybu.04", 0x080000, 0x80000, 0x1eec68ac )
	ROM_LOAD16_WORD_SWAP( "cyb.05",  0x100000, 0x80000, 0xec40408e )
	ROM_LOAD16_WORD_SWAP( "cyb.06",  0x180000, 0x80000, 0x1ad0bed2 )
	ROM_LOAD16_WORD_SWAP( "cyb.07",  0x200000, 0x80000, 0x6245a39a )
	ROM_LOAD16_WORD_SWAP( "cyb.08",  0x280000, 0x80000, 0x4b48e223 )
	ROM_LOAD16_WORD_SWAP( "cyb.09",  0x300000, 0x80000, 0xe15238f6 )
	ROM_LOAD16_WORD_SWAP( "cyb.10",  0x380000, 0x80000, 0x75f4003b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "cyb.13m",   0x0000000, 0x400000, 0xf0dce192 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.15m",   0x0000002, 0x400000, 0x187aa39c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.17m",   0x0000004, 0x400000, 0x8a0e4b12 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.19m",   0x0000006, 0x400000, 0x34b62612 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.14m",   0x1000000, 0x400000, 0xc1537957 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.16m",   0x1000002, 0x400000, 0x15349e86 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.18m",   0x1000004, 0x400000, 0xd83e977d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.20m",   0x1000006, 0x400000, 0x77cdad5c , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "cyb.01",   0x00000, 0x08000, 0x9c0fb079 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "cyb.02",   0x28000, 0x20000, 0x51cb0c4e )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "cyb.11m",   0x000000, 0x200000, 0x362ccab2 )
	ROM_LOAD16_WORD_SWAP( "cyb.12m",   0x200000, 0x200000, 0x7066e9cc )
ROM_END

ROM_START( cybotsj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "cybj.03", 0x000000, 0x80000, 0x6096eada )
	ROM_LOAD16_WORD_SWAP( "cybj.04", 0x080000, 0x80000, 0x7b0ffaa9 )
	ROM_LOAD16_WORD_SWAP( "cyb.05",  0x100000, 0x80000, 0xec40408e )
	ROM_LOAD16_WORD_SWAP( "cyb.06",  0x180000, 0x80000, 0x1ad0bed2 )
	ROM_LOAD16_WORD_SWAP( "cyb.07",  0x200000, 0x80000, 0x6245a39a )
	ROM_LOAD16_WORD_SWAP( "cyb.08",  0x280000, 0x80000, 0x4b48e223 )
	ROM_LOAD16_WORD_SWAP( "cyb.09",  0x300000, 0x80000, 0xe15238f6 )
	ROM_LOAD16_WORD_SWAP( "cyb.10",  0x380000, 0x80000, 0x75f4003b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "cyb.13m",   0x0000000, 0x400000, 0xf0dce192 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.15m",   0x0000002, 0x400000, 0x187aa39c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.17m",   0x0000004, 0x400000, 0x8a0e4b12 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.19m",   0x0000006, 0x400000, 0x34b62612 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.14m",   0x1000000, 0x400000, 0xc1537957 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.16m",   0x1000002, 0x400000, 0x15349e86 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.18m",   0x1000004, 0x400000, 0xd83e977d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "cyb.20m",   0x1000006, 0x400000, 0x77cdad5c , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "cyb.01",   0x00000, 0x08000, 0x9c0fb079 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "cyb.02",   0x28000, 0x20000, 0x51cb0c4e )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "cyb.11m",   0x000000, 0x200000, 0x362ccab2 )
	ROM_LOAD16_WORD_SWAP( "cyb.12m",   0x200000, 0x200000, 0x7066e9cc )
ROM_END

ROM_START( ddtod )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dade.03c", 0x000000, 0x80000, 0x8e73533d )
	ROM_LOAD16_WORD_SWAP( "dade.04c", 0x080000, 0x80000, 0x00c2e82e )
	ROM_LOAD16_WORD_SWAP( "dade.05c", 0x100000, 0x80000, 0xea996008 )
	ROM_LOAD16_WORD_SWAP( "dad.06a",  0x180000, 0x80000, 0x6225495a )
	ROM_LOAD16_WORD_SWAP( "dad.07a",  0x200000, 0x80000, 0xb3480ec3 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "dad.13m",   0x000000, 0x200000, 0xda3cb7d6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.15m",   0x000002, 0x200000, 0x92b63172 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.17m",   0x000004, 0x200000, 0xb98757f5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.19m",   0x000006, 0x200000, 0x8121ce46 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.14m",   0x800000, 0x100000, 0x837e6f3f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.16m",   0x800002, 0x100000, 0xf0916bdb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.18m",   0x800004, 0x100000, 0xcef393ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.20m",   0x800006, 0x100000, 0x8953fe9e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dad.01",   0x00000, 0x08000, 0x3f5e2424 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dad.11m",   0x000000, 0x200000, 0x0c499b67 )
	ROM_LOAD16_WORD_SWAP( "dad.12m",   0x200000, 0x200000, 0x2f0b5a4e )
ROM_END

ROM_START( ddtodr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dade.03a", 0x000000, 0x80000, 0x665a035e )
	ROM_LOAD16_WORD_SWAP( "dade.04a", 0x080000, 0x80000, 0x02613207 )
	ROM_LOAD16_WORD_SWAP( "dade.05a", 0x100000, 0x80000, 0x36845996 )
	ROM_LOAD16_WORD_SWAP( "dad.06",   0x180000, 0x80000, 0x13aa3e56 )
	ROM_LOAD16_WORD_SWAP( "dad.07",   0x200000, 0x80000, 0x431cb6dd )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "dad.13m",   0x000000, 0x200000, 0xda3cb7d6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.15m",   0x000002, 0x200000, 0x92b63172 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.17m",   0x000004, 0x200000, 0xb98757f5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.19m",   0x000006, 0x200000, 0x8121ce46 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.14m",   0x800000, 0x100000, 0x837e6f3f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.16m",   0x800002, 0x100000, 0xf0916bdb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.18m",   0x800004, 0x100000, 0xcef393ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.20m",   0x800006, 0x100000, 0x8953fe9e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dad.01",   0x00000, 0x08000, 0x3f5e2424 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dad.11m",   0x000000, 0x200000, 0x0c499b67 )
	ROM_LOAD16_WORD_SWAP( "dad.12m",   0x200000, 0x200000, 0x2f0b5a4e )
ROM_END

ROM_START( ddtodu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dadu.03b", 0x000000, 0x80000, 0xa519905f )
	ROM_LOAD16_WORD_SWAP( "dadu.04b", 0x080000, 0x80000, 0x52562d38 )
	ROM_LOAD16_WORD_SWAP( "dadu.05b", 0x100000, 0x80000, 0xee1cfbfe )
	ROM_LOAD16_WORD_SWAP( "dad.06",   0x180000, 0x80000, 0x13aa3e56 )
	ROM_LOAD16_WORD_SWAP( "dad.07",   0x200000, 0x80000, 0x431cb6dd )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "dad.13m",   0x000000, 0x200000, 0xda3cb7d6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.15m",   0x000002, 0x200000, 0x92b63172 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.17m",   0x000004, 0x200000, 0xb98757f5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.19m",   0x000006, 0x200000, 0x8121ce46 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.14m",   0x800000, 0x100000, 0x837e6f3f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.16m",   0x800002, 0x100000, 0xf0916bdb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.18m",   0x800004, 0x100000, 0xcef393ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.20m",   0x800006, 0x100000, 0x8953fe9e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dad.01",   0x00000, 0x08000, 0x3f5e2424 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dad.11m",   0x000000, 0x200000, 0x0c499b67 )
	ROM_LOAD16_WORD_SWAP( "dad.12m",   0x200000, 0x200000, 0x2f0b5a4e )
ROM_END

ROM_START( ddtodur1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
  	ROM_LOAD16_WORD_SWAP( "dadu.03a", 0x000000, 0x80000, 0x4413f177 )
	ROM_LOAD16_WORD_SWAP( "dadu.04a", 0x080000, 0x80000, 0x168de230 )
	ROM_LOAD16_WORD_SWAP( "dadu.05a", 0x100000, 0x80000, 0x03d39e91 )
	ROM_LOAD16_WORD_SWAP( "dad.06",   0x180000, 0x80000, 0x13aa3e56 )
	ROM_LOAD16_WORD_SWAP( "dad.07",   0x200000, 0x80000, 0x431cb6dd )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "dad.13m",   0x000000, 0x200000, 0xda3cb7d6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.15m",   0x000002, 0x200000, 0x92b63172 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.17m",   0x000004, 0x200000, 0xb98757f5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.19m",   0x000006, 0x200000, 0x8121ce46 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.14m",   0x800000, 0x100000, 0x837e6f3f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.16m",   0x800002, 0x100000, 0xf0916bdb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.18m",   0x800004, 0x100000, 0xcef393ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.20m",   0x800006, 0x100000, 0x8953fe9e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dad.01",   0x00000, 0x08000, 0x3f5e2424 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dad.11m",   0x000000, 0x200000, 0x0c499b67 )
	ROM_LOAD16_WORD_SWAP( "dad.12m",   0x200000, 0x200000, 0x2f0b5a4e )
ROM_END

ROM_START( ddtodj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dadj.03c", 0x000000, 0x80000, 0x0b1b5798 )
	ROM_LOAD16_WORD_SWAP( "dadj.04c", 0x080000, 0x80000, 0xc6a2fbc8 )
	ROM_LOAD16_WORD_SWAP( "dadj.05c", 0x100000, 0x80000, 0x189b15fe )
	ROM_LOAD16_WORD_SWAP( "dad.06a",  0x180000, 0x80000, 0x6225495a )
	ROM_LOAD16_WORD_SWAP( "dad.07a",  0x200000, 0x80000, 0xb3480ec3 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "dad.13m",   0x000000, 0x200000, 0xda3cb7d6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.15m",   0x000002, 0x200000, 0x92b63172 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.17m",   0x000004, 0x200000, 0xb98757f5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.19m",   0x000006, 0x200000, 0x8121ce46 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.14m",   0x800000, 0x100000, 0x837e6f3f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.16m",   0x800002, 0x100000, 0xf0916bdb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.18m",   0x800004, 0x100000, 0xcef393ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.20m",   0x800006, 0x100000, 0x8953fe9e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (banks) */
	ROM_LOAD( "dad.01",   0x00000, 0x08000, 0x3f5e2424 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dad.11m",   0x000000, 0x200000, 0x0c499b67 )
	ROM_LOAD16_WORD_SWAP( "dad.12m",   0x200000, 0x200000, 0x2f0b5a4e )
ROM_END

ROM_START( ddtodjr2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dadj.03b", 0x000000, 0x80000, 0x87606b85 )
	ROM_LOAD16_WORD_SWAP( "dadj.04b", 0x080000, 0x80000, 0x24d49575 )
	ROM_LOAD16_WORD_SWAP( "dadj.05b", 0x100000, 0x80000, 0x56ce51f7 )
	ROM_LOAD16_WORD_SWAP( "dad.06",   0x180000, 0x80000, 0x13aa3e56 )
	ROM_LOAD16_WORD_SWAP( "dad.07",   0x200000, 0x80000, 0x431cb6dd )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "dad.13m",   0x000000, 0x200000, 0xda3cb7d6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.15m",   0x000002, 0x200000, 0x92b63172 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.17m",   0x000004, 0x200000, 0xb98757f5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.19m",   0x000006, 0x200000, 0x8121ce46 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.14m",   0x800000, 0x100000, 0x837e6f3f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.16m",   0x800002, 0x100000, 0xf0916bdb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.18m",   0x800004, 0x100000, 0xcef393ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.20m",   0x800006, 0x100000, 0x8953fe9e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dad.01",   0x00000, 0x08000, 0x3f5e2424 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dad.11m",   0x000000, 0x200000, 0x0c499b67 )
	ROM_LOAD16_WORD_SWAP( "dad.12m",   0x200000, 0x200000, 0x2f0b5a4e )
ROM_END

ROM_START( ddtodjr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dadj.03a", 0x000000, 0x80000, 0x711638dc )
	ROM_LOAD16_WORD_SWAP( "dadj.04a", 0x080000, 0x80000, 0x4869639c )
	ROM_LOAD16_WORD_SWAP( "dadj.05a", 0x100000, 0x80000, 0x484c0efa )
	ROM_LOAD16_WORD_SWAP( "dad.06",   0x180000, 0x80000, 0x13aa3e56 )
	ROM_LOAD16_WORD_SWAP( "dad.07",   0x200000, 0x80000, 0x431cb6dd )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "dad.13m",   0x000000, 0x200000, 0xda3cb7d6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.15m",   0x000002, 0x200000, 0x92b63172 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.17m",   0x000004, 0x200000, 0xb98757f5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.19m",   0x000006, 0x200000, 0x8121ce46 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.14m",   0x800000, 0x100000, 0x837e6f3f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.16m",   0x800002, 0x100000, 0xf0916bdb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.18m",   0x800004, 0x100000, 0xcef393ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.20m",   0x800006, 0x100000, 0x8953fe9e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dad.01",   0x00000, 0x08000, 0x3f5e2424 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dad.11m",   0x000000, 0x200000, 0x0c499b67 )
	ROM_LOAD16_WORD_SWAP( "dad.12m",   0x200000, 0x200000, 0x2f0b5a4e )
ROM_END

ROM_START( ddtoda )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dada.03a", 0x000000, 0x80000, 0xfc6f2dd7 )
	ROM_LOAD16_WORD_SWAP( "dada.04a", 0x080000, 0x80000, 0xd4be4009 )
	ROM_LOAD16_WORD_SWAP( "dada.05a", 0x100000, 0x80000, 0x6712d1cf )
	ROM_LOAD16_WORD_SWAP( "dad.06",   0x180000, 0x80000, 0x13aa3e56 )
	ROM_LOAD16_WORD_SWAP( "dad.07",   0x200000, 0x80000, 0x431cb6dd )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "dad.13m",   0x000000, 0x200000, 0xda3cb7d6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.15m",   0x000002, 0x200000, 0x92b63172 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.17m",   0x000004, 0x200000, 0xb98757f5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.19m",   0x000006, 0x200000, 0x8121ce46 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.14m",   0x800000, 0x100000, 0x837e6f3f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.16m",   0x800002, 0x100000, 0xf0916bdb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.18m",   0x800004, 0x100000, 0xcef393ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.20m",   0x800006, 0x100000, 0x8953fe9e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dad.01",   0x00000, 0x08000, 0x3f5e2424 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dad.11m",   0x000000, 0x200000, 0x0c499b67 )
	ROM_LOAD16_WORD_SWAP( "dad.12m",   0x200000, 0x200000, 0x2f0b5a4e )
ROM_END

ROM_START( ddtodh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dadh.03b", 0x000000, 0x80000, 0xae0cb98e )
	ROM_LOAD16_WORD_SWAP( "dadh.04b", 0x080000, 0x80000, 0xb5774363 )
	ROM_LOAD16_WORD_SWAP( "dadh.05b", 0x100000, 0x80000, 0x6ce2a485 )
	ROM_LOAD16_WORD_SWAP( "dad.06",   0x180000, 0x80000, 0x13aa3e56 )
	ROM_LOAD16_WORD_SWAP( "dad.07",   0x200000, 0x80000, 0x431cb6dd )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "dad.13m",   0x000000, 0x200000, 0xda3cb7d6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.15m",   0x000002, 0x200000, 0x92b63172 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.17m",   0x000004, 0x200000, 0xb98757f5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.19m",   0x000006, 0x200000, 0x8121ce46 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.14m",   0x800000, 0x100000, 0x837e6f3f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.16m",   0x800002, 0x100000, 0xf0916bdb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.18m",   0x800004, 0x100000, 0xcef393ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dad.20m",   0x800006, 0x100000, 0x8953fe9e , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dad.01",   0x00000, 0x08000, 0x3f5e2424 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dad.11m",   0x000000, 0x200000, 0x0c499b67 )
	ROM_LOAD16_WORD_SWAP( "dad.12m",   0x200000, 0x200000, 0x2f0b5a4e )
ROM_END

ROM_START( ddsom )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2e.03e", 0x000000, 0x80000, 0x449361af )
	ROM_LOAD16_WORD_SWAP( "dd2e.04e", 0x080000, 0x80000, 0x5b7052b6 )
	ROM_LOAD16_WORD_SWAP( "dd2e.05e", 0x100000, 0x80000, 0x788d5f60 )
	ROM_LOAD16_WORD_SWAP( "dd2e.06e", 0x180000, 0x80000, 0xe0807e1e )
	ROM_LOAD16_WORD_SWAP( "dd2e.07",  0x200000, 0x80000, 0xbb777a02 )
	ROM_LOAD16_WORD_SWAP( "dd2e.08",  0x280000, 0x80000, 0x30970890 )
	ROM_LOAD16_WORD_SWAP( "dd2e.09",  0x300000, 0x80000, 0x99e2194d )
	ROM_LOAD16_WORD_SWAP( "dd2e.10",  0x380000, 0x80000, 0xe198805e )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( ddsomr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2e.03d", 0x000000, 0x80000, 0x6c084ab5 )
	ROM_LOAD16_WORD_SWAP( "dd2e.04d", 0x080000, 0x80000, 0x9b94a947 )
	ROM_LOAD16_WORD_SWAP( "dd2e.05d", 0x100000, 0x80000, 0x5d6a63c6 )
	ROM_LOAD16_WORD_SWAP( "dd2e.06d", 0x180000, 0x80000, 0x31bde8ee )
	ROM_LOAD16_WORD_SWAP( "dd2e.07",  0x200000, 0x80000, 0xbb777a02 )
	ROM_LOAD16_WORD_SWAP( "dd2e.08",  0x280000, 0x80000, 0x30970890 )
	ROM_LOAD16_WORD_SWAP( "dd2e.09",  0x300000, 0x80000, 0x99e2194d )
	ROM_LOAD16_WORD_SWAP( "dd2e.10",  0x380000, 0x80000, 0xe198805e )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( ddsomr2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2e.03b", 0x000000, 0x80000, 0xcd2deb66 )
	ROM_LOAD16_WORD_SWAP( "dd2e.04b", 0x080000, 0x80000, 0xbfee43cc )
	ROM_LOAD16_WORD_SWAP( "dd2e.05b", 0x100000, 0x80000, 0x049ab19d )
	ROM_LOAD16_WORD_SWAP( "dd2e.06b", 0x180000, 0x80000, 0x3994fb8b )
	ROM_LOAD16_WORD_SWAP( "dd2e.07",  0x200000, 0x80000, 0xbb777a02 )
	ROM_LOAD16_WORD_SWAP( "dd2e.08",  0x280000, 0x80000, 0x30970890 )
	ROM_LOAD16_WORD_SWAP( "dd2e.09",  0x300000, 0x80000, 0x99e2194d )
	ROM_LOAD16_WORD_SWAP( "dd2e.10",  0x380000, 0x80000, 0xe198805e )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( ddsomr3 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2e.03a", 0x000000, 0x80000, 0x6de67678 )
	ROM_LOAD16_WORD_SWAP( "dd2e.04a", 0x080000, 0x80000, 0x0e45739a )
	ROM_LOAD16_WORD_SWAP( "dd2e.05a", 0x100000, 0x80000, 0x3dce8025 )
	ROM_LOAD16_WORD_SWAP( "dd2e.06a", 0x180000, 0x80000, 0x51bafbef )
	ROM_LOAD16_WORD_SWAP( "dd2e.07",  0x200000, 0x80000, 0xbb777a02 )
	ROM_LOAD16_WORD_SWAP( "dd2e.08",  0x280000, 0x80000, 0x30970890 )
	ROM_LOAD16_WORD_SWAP( "dd2e.09",  0x300000, 0x80000, 0x99e2194d )
	ROM_LOAD16_WORD_SWAP( "dd2e.10",  0x380000, 0x80000, 0xe198805e )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( ddsomu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2u.03g", 0x000000, 0x80000, 0xfb089b39 )
	ROM_LOAD16_WORD_SWAP( "dd2u.04g", 0x080000, 0x80000, 0xcd432b73 )
	ROM_LOAD16_WORD_SWAP( "dd2.05g",  0x100000, 0x80000, 0x5eb1991c )
	ROM_LOAD16_WORD_SWAP( "dd2.06g",  0x180000, 0x80000, 0xc26b5e55 )
	ROM_LOAD16_WORD_SWAP( "dd2.07",   0x200000, 0x80000, 0x909a0b8b )
	ROM_LOAD16_WORD_SWAP( "dd2.08",   0x280000, 0x80000, 0xe53c4d01 )
	ROM_LOAD16_WORD_SWAP( "dd2.09",   0x300000, 0x80000, 0x5f86279f )
	ROM_LOAD16_WORD_SWAP( "dd2.10",   0x380000, 0x80000, 0xad954c26 )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( ddsomur1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2u.03d", 0x000000, 0x80000, 0x0f700d84 )
	ROM_LOAD16_WORD_SWAP( "dd2u.04d", 0x080000, 0x80000, 0xb99eb254 )
	ROM_LOAD16_WORD_SWAP( "dd2.05d",  0x100000, 0x80000, 0xb23061f3 )
	ROM_LOAD16_WORD_SWAP( "dd2.06d",  0x180000, 0x80000, 0x8bf1d8ce )
	ROM_LOAD16_WORD_SWAP( "dd2.07",   0x200000, 0x80000, 0x909a0b8b )
	ROM_LOAD16_WORD_SWAP( "dd2.08",   0x280000, 0x80000, 0xe53c4d01 )
	ROM_LOAD16_WORD_SWAP( "dd2.09",   0x300000, 0x80000, 0x5f86279f )
	ROM_LOAD16_WORD_SWAP( "dd2.10",   0x380000, 0x80000, 0xad954c26 )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( ddsomj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2j.03g", 0x000000, 0x80000, 0xe6c8c985 )
	ROM_LOAD16_WORD_SWAP( "dd2j.04g", 0x080000, 0x80000, 0x8386c0bd )
	ROM_LOAD16_WORD_SWAP( "dd2.05g",  0x100000, 0x80000, 0x5eb1991c )
	ROM_LOAD16_WORD_SWAP( "dd2.06g",  0x180000, 0x80000, 0xc26b5e55 )
	ROM_LOAD16_WORD_SWAP( "dd2.07",   0x200000, 0x80000, 0x909a0b8b )
	ROM_LOAD16_WORD_SWAP( "dd2.08",   0x280000, 0x80000, 0xe53c4d01 )
	ROM_LOAD16_WORD_SWAP( "dd2.09",   0x300000, 0x80000, 0x5f86279f )
	ROM_LOAD16_WORD_SWAP( "dd2.10",   0x380000, 0x80000, 0xad954c26 )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( ddsomjr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2j.03b", 0x000000, 0x80000, 0x965d74e5 )
	ROM_LOAD16_WORD_SWAP( "dd2j.04b", 0x080000, 0x80000, 0x958eb8f3 )
	ROM_LOAD16_WORD_SWAP( "dd2.05b",  0x100000, 0x80000, 0xd38571ca )
	ROM_LOAD16_WORD_SWAP( "dd2.06b",  0x180000, 0x80000, 0x6d5a3bbb )
	ROM_LOAD16_WORD_SWAP( "dd2.07",   0x200000, 0x80000, 0x909a0b8b )
	ROM_LOAD16_WORD_SWAP( "dd2.08",   0x280000, 0x80000, 0xe53c4d01 )
	ROM_LOAD16_WORD_SWAP( "dd2.09",   0x300000, 0x80000, 0x5f86279f )
	ROM_LOAD16_WORD_SWAP( "dd2.10",   0x380000, 0x80000, 0xad954c26 )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( ddsoma )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2a.03g", 0x000000, 0x80000, 0x0b4fec22 )
	ROM_LOAD16_WORD_SWAP( "dd2a.04g", 0x080000, 0x80000, 0x055b7019 )
	ROM_LOAD16_WORD_SWAP( "dd2.05g",  0x100000, 0x80000, 0x5eb1991c )
	ROM_LOAD16_WORD_SWAP( "dd2.06g",  0x180000, 0x80000, 0xc26b5e55 )
	ROM_LOAD16_WORD_SWAP( "dd2.07",   0x200000, 0x80000, 0x909a0b8b )
	ROM_LOAD16_WORD_SWAP( "dd2.08",   0x280000, 0x80000, 0xe53c4d01 )
	ROM_LOAD16_WORD_SWAP( "dd2.09",   0x300000, 0x80000, 0x5f86279f )
	ROM_LOAD16_WORD_SWAP( "dd2.10",   0x380000, 0x80000, 0xad954c26 )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( ddsomb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "dd2b.03a", 0x000000, 0x80000, 0xe8ce7fbb )
	ROM_LOAD16_WORD_SWAP( "dd2b.04a", 0x080000, 0x80000, 0x6b679664 )
	ROM_LOAD16_WORD_SWAP( "dd2b.05a", 0x100000, 0x80000, 0x9b2534eb )
	ROM_LOAD16_WORD_SWAP( "dd2b.06a", 0x180000, 0x80000, 0x3b21ba59 )
	ROM_LOAD16_WORD_SWAP( "dd2b.07",  0x200000, 0x80000, 0xfce2558d )
	ROM_LOAD16_WORD_SWAP( "dd2e.08",  0x280000, 0x80000, 0x30970890 )
	ROM_LOAD16_WORD_SWAP( "dd2e.09",  0x300000, 0x80000, 0x99e2194d )
	ROM_LOAD16_WORD_SWAP( "dd2e.10",  0x380000, 0x80000, 0xe198805e )

	ROM_REGION( 0x1800000, REGION_GFX1, 0 )
	ROMX_LOAD( "dd2.13m",   0x0000000, 0x400000, 0xa46b4e6e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.15m",   0x0000002, 0x400000, 0xd5fc50fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.17m",   0x0000004, 0x400000, 0x837c0867 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.19m",   0x0000006, 0x400000, 0xbb0ec21c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.14m",   0x1000000, 0x200000, 0x6d824ce2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.16m",   0x1000002, 0x200000, 0x79682ae5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.18m",   0x1000004, 0x200000, 0xacddd149 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "dd2.20m",   0x1000006, 0x200000, 0x117fb0c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "dd2.01",   0x00000, 0x08000, 0x99d657e5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "dd2.02",   0x28000, 0x20000, 0x117a3824 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "dd2.11m",   0x000000, 0x200000, 0x98d0c325 )
	ROM_LOAD16_WORD_SWAP( "dd2.12m",   0x200000, 0x200000, 0x5ea2e7fa )
ROM_END

ROM_START( dimahoo )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "gdme_03.03", 0x000000, 0x80000, 0x968fcecd )
	ROM_LOAD16_WORD_SWAP( "gmd.04",  0x080000, 0x80000, 0x37485567 )
	ROM_LOAD16_WORD_SWAP( "gmd.05",  0x100000, 0x80000, 0xda269ffb )
	ROM_LOAD16_WORD_SWAP( "gmd.06",  0x180000, 0x80000, 0x55b483c9 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "gmd.13m",   0x000000, 0x400000, 0x80dd19f0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "gmd.15m",   0x000002, 0x400000, 0xdfd93a78 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "gmd.17m",   0x000004, 0x400000, 0x16356520 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "gmd.19m",   0x000006, 0x400000, 0xdfc33031 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "gmd.01",   0x00000, 0x08000, 0x3f9bc985 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "gmd.02",   0x28000, 0x20000, 0x3fd39dde )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "gmd.11m",   0x000000, 0x400000, 0x06a65542 )
	ROM_LOAD16_WORD_SWAP( "gmd.12m",   0x400000, 0x400000, 0x50bc7a31 )
ROM_END

ROM_START( dimahoou )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "gmdu.03", 0x000000, 0x80000, 0x43bcb15f )
	ROM_LOAD16_WORD_SWAP( "gmd.04",  0x080000, 0x80000, 0x37485567 )
	ROM_LOAD16_WORD_SWAP( "gmd.05",  0x100000, 0x80000, 0xda269ffb )
	ROM_LOAD16_WORD_SWAP( "gmd.06",  0x180000, 0x80000, 0x55b483c9 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "gmd.13m",   0x000000, 0x400000, 0x80dd19f0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "gmd.15m",   0x000002, 0x400000, 0xdfd93a78 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "gmd.17m",   0x000004, 0x400000, 0x16356520 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "gmd.19m",   0x000006, 0x400000, 0xdfc33031 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "gmd.01",   0x00000, 0x08000, 0x3f9bc985 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "gmd.02",   0x28000, 0x20000, 0x3fd39dde )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "gmd.11m",   0x000000, 0x400000, 0x06a65542 )
	ROM_LOAD16_WORD_SWAP( "gmd.12m",   0x400000, 0x400000, 0x50bc7a31 )
ROM_END

ROM_START( gmahou )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "gmdj.03", 0x000000, 0x80000, 0xcd6979e3 )
	ROM_LOAD16_WORD_SWAP( "gmd.04",  0x080000, 0x80000, 0x37485567 )
	ROM_LOAD16_WORD_SWAP( "gmd.05",  0x100000, 0x80000, 0xda269ffb )
	ROM_LOAD16_WORD_SWAP( "gmd.06",  0x180000, 0x80000, 0x55b483c9 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "gmd.13m",   0x000000, 0x400000, 0x80dd19f0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "gmd.15m",   0x000002, 0x400000, 0xdfd93a78 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "gmd.17m",   0x000004, 0x400000, 0x16356520 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "gmd.19m",   0x000006, 0x400000, 0xdfc33031 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "gmd.01",   0x00000, 0x08000, 0x3f9bc985 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "gmd.02",   0x28000, 0x20000, 0x3fd39dde )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "gmd.11m",   0x000000, 0x400000, 0x06a65542 )
	ROM_LOAD16_WORD_SWAP( "gmd.12m",   0x400000, 0x400000, 0x50bc7a31 )
ROM_END

ROM_START( dstlk )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vame.03a", 0x000000, 0x80000, 0x004c9cff )
	ROM_LOAD16_WORD_SWAP( "vame.04a", 0x080000, 0x80000, 0xae413ff2 )
	ROM_LOAD16_WORD_SWAP( "vame.05a", 0x100000, 0x80000, 0x60678756 )
	ROM_LOAD16_WORD_SWAP( "vame.06a", 0x180000, 0x80000, 0x912870b3 )
	ROM_LOAD16_WORD_SWAP( "vame.07a", 0x200000, 0x80000, 0xdabae3e8 )
	ROM_LOAD16_WORD_SWAP( "vame.08a", 0x280000, 0x80000, 0x2c6e3077 )
	ROM_LOAD16_WORD_SWAP( "vame.09a", 0x300000, 0x80000, 0xf16db74b )
	ROM_LOAD16_WORD_SWAP( "vame.10a", 0x380000, 0x80000, 0x701e2147 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "vam.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.14m",   0x1000000, 0x100000, 0xbd87243c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.16m",   0x1000002, 0x100000, 0xafec855f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.18m",   0x1000004, 0x100000, 0x3a033625 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.20m",   0x1000006, 0x100000, 0x2bff6a89 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vam.01",   0x00000, 0x08000, 0x64b685d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vam.02",   0x28000, 0x20000, 0xcf7c97c7 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vam.11m",   0x000000, 0x200000, 0x4a39deb2 )
	ROM_LOAD16_WORD_SWAP( "vam.12m",   0x200000, 0x200000, 0x1a3e5c03 )
ROM_END

ROM_START( dstlku )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vamu.03b", 0x000000, 0x80000, 0x68a6343f )
	ROM_LOAD16_WORD_SWAP( "vamu.04b", 0x080000, 0x80000, 0x58161453 )
	ROM_LOAD16_WORD_SWAP( "vamu.05b", 0x100000, 0x80000, 0xdfc038b8 )
	ROM_LOAD16_WORD_SWAP( "vamu.06b", 0x180000, 0x80000, 0xc3842c89 )
	ROM_LOAD16_WORD_SWAP( "vamu.07b", 0x200000, 0x80000, 0x25b60b6e )
	ROM_LOAD16_WORD_SWAP( "vamu.08b", 0x280000, 0x80000, 0x2113c596 )
	ROM_LOAD16_WORD_SWAP( "vamu.09b", 0x300000, 0x80000, 0x2d1e9ae5 )
	ROM_LOAD16_WORD_SWAP( "vamu.10b", 0x380000, 0x80000, 0x81145622 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "vam.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.14m",   0x1000000, 0x100000, 0xbd87243c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.16m",   0x1000002, 0x100000, 0xafec855f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.18m",   0x1000004, 0x100000, 0x3a033625 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.20m",   0x1000006, 0x100000, 0x2bff6a89 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vam.01",   0x00000, 0x08000, 0x64b685d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vam.02",   0x28000, 0x20000, 0xcf7c97c7 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vam.11m",   0x000000, 0x200000, 0x4a39deb2 )
	ROM_LOAD16_WORD_SWAP( "vam.12m",   0x200000, 0x200000, 0x1a3e5c03 )
ROM_END

ROM_START( dstlkur1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vamu.03a", 0x000000, 0x80000, 0x628899f9 )
	ROM_LOAD16_WORD_SWAP( "vamu.04a", 0x080000, 0x80000, 0x696d9b25 )
	ROM_LOAD16_WORD_SWAP( "vamu.05a", 0x100000, 0x80000, 0x673ed50a )
	ROM_LOAD16_WORD_SWAP( "vamu.06a", 0x180000, 0x80000, 0xf2377be7 )
	ROM_LOAD16_WORD_SWAP( "vamu.07a", 0x200000, 0x80000, 0xd8f498c4 )
	ROM_LOAD16_WORD_SWAP( "vamu.08a", 0x280000, 0x80000, 0xe6a8a1a0 )
	ROM_LOAD16_WORD_SWAP( "vamu.09a", 0x300000, 0x80000, 0x8dd55b24 )
	ROM_LOAD16_WORD_SWAP( "vamu.10a", 0x380000, 0x80000, 0xc1a3d9be )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "vam.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.14m",   0x1000000, 0x100000, 0xbd87243c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.16m",   0x1000002, 0x100000, 0xafec855f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.18m",   0x1000004, 0x100000, 0x3a033625 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.20m",   0x1000006, 0x100000, 0x2bff6a89 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vam.01",   0x00000, 0x08000, 0x64b685d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vam.02",   0x28000, 0x20000, 0xcf7c97c7 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vam.11m",   0x000000, 0x200000, 0x4a39deb2 )
	ROM_LOAD16_WORD_SWAP( "vam.12m",   0x200000, 0x200000, 0x1a3e5c03 )
ROM_END

ROM_START( vampj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vamj.03a", 0x000000, 0x80000, 0xf55d3722 )
	ROM_LOAD16_WORD_SWAP( "vamj.04b", 0x080000, 0x80000, 0x4d9c43c4 )
	ROM_LOAD16_WORD_SWAP( "vamj.05a", 0x100000, 0x80000, 0x6c497e92 )
	ROM_LOAD16_WORD_SWAP( "vamj.06a", 0x180000, 0x80000, 0xf1bbecb6 )
	ROM_LOAD16_WORD_SWAP( "vamj.07a", 0x200000, 0x80000, 0x1067ad84 )
	ROM_LOAD16_WORD_SWAP( "vamj.08a", 0x280000, 0x80000, 0x4b89f41f )
	ROM_LOAD16_WORD_SWAP( "vamj.09a", 0x300000, 0x80000, 0xfc0a4aac )
	ROM_LOAD16_WORD_SWAP( "vamj.10a", 0x380000, 0x80000, 0x9270c26b )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "vam.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.14m",   0x1000000, 0x100000, 0xbd87243c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.16m",   0x1000002, 0x100000, 0xafec855f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.18m",   0x1000004, 0x100000, 0x3a033625 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.20m",   0x1000006, 0x100000, 0x2bff6a89 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vam.01",   0x00000, 0x08000, 0x64b685d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vam.02",   0x28000, 0x20000, 0xcf7c97c7 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vam.11m",   0x000000, 0x200000, 0x4a39deb2 )
	ROM_LOAD16_WORD_SWAP( "vam.12m",   0x200000, 0x200000, 0x1a3e5c03 )
ROM_END

ROM_START( vampja )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vamj.03a", 0x000000, 0x80000, 0xf55d3722 )
	ROM_LOAD16_WORD_SWAP( "vamj.04a", 0x080000, 0x80000, 0xfdcbdae3 )
	ROM_LOAD16_WORD_SWAP( "vamj.05a", 0x100000, 0x80000, 0x6c497e92 )
	ROM_LOAD16_WORD_SWAP( "vamj.06a", 0x180000, 0x80000, 0xf1bbecb6 )
	ROM_LOAD16_WORD_SWAP( "vamj.07a", 0x200000, 0x80000, 0x1067ad84 )
	ROM_LOAD16_WORD_SWAP( "vamj.08a", 0x280000, 0x80000, 0x4b89f41f )
	ROM_LOAD16_WORD_SWAP( "vamj.09a", 0x300000, 0x80000, 0xfc0a4aac )
	ROM_LOAD16_WORD_SWAP( "vamj.10a", 0x380000, 0x80000, 0x9270c26b )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "vam.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.14m",   0x1000000, 0x100000, 0xbd87243c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.16m",   0x1000002, 0x100000, 0xafec855f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.18m",   0x1000004, 0x100000, 0x3a033625 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.20m",   0x1000006, 0x100000, 0x2bff6a89 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vam.01",   0x00000, 0x08000, 0x64b685d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vam.02",   0x28000, 0x20000, 0xcf7c97c7 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vam.11m",   0x000000, 0x200000, 0x4a39deb2 )
	ROM_LOAD16_WORD_SWAP( "vam.12m",   0x200000, 0x200000, 0x1a3e5c03 )
ROM_END

ROM_START( vampjr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vamj.03", 0x000000, 0x80000, 0x8895bf77 )
	ROM_LOAD16_WORD_SWAP( "vamj.04", 0x080000, 0x80000, 0x5027db3d )
	ROM_LOAD16_WORD_SWAP( "vamj.05", 0x100000, 0x80000, 0x97c66fdb )
	ROM_LOAD16_WORD_SWAP( "vamj.06", 0x180000, 0x80000, 0x9b4c3426 )
	ROM_LOAD16_WORD_SWAP( "vamj.07", 0x200000, 0x80000, 0x303bc4fd )
	ROM_LOAD16_WORD_SWAP( "vamj.08", 0x280000, 0x80000, 0x3dea3646 )
	ROM_LOAD16_WORD_SWAP( "vamj.09", 0x300000, 0x80000, 0xc119a827 )
	ROM_LOAD16_WORD_SWAP( "vamj.10", 0x380000, 0x80000, 0x46593b79 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "vam.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.14m",   0x1000000, 0x100000, 0xbd87243c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.16m",   0x1000002, 0x100000, 0xafec855f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.18m",   0x1000004, 0x100000, 0x3a033625 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.20m",   0x1000006, 0x100000, 0x2bff6a89 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vam.01",   0x00000, 0x08000, 0x64b685d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vam.02",   0x28000, 0x20000, 0xcf7c97c7 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vam.11m",   0x000000, 0x200000, 0x4a39deb2 )
	ROM_LOAD16_WORD_SWAP( "vam.12m",   0x200000, 0x200000, 0x1a3e5c03 )
ROM_END

ROM_START( dstlka )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vama.03a", 0x000000, 0x80000, 0x294e0bec )
	ROM_LOAD16_WORD_SWAP( "vama.04a", 0x080000, 0x80000, 0xbc18e128 )
	ROM_LOAD16_WORD_SWAP( "vama.05a", 0x100000, 0x80000, 0xe709fa59 )
	ROM_LOAD16_WORD_SWAP( "vama.06a", 0x180000, 0x80000, 0x55e4d387 )
	ROM_LOAD16_WORD_SWAP( "vama.07a", 0x200000, 0x80000, 0x24e8f981 )
	ROM_LOAD16_WORD_SWAP( "vama.08a", 0x280000, 0x80000, 0x743f3a8e )
	ROM_LOAD16_WORD_SWAP( "vama.09a", 0x300000, 0x80000, 0x67fa5573 )
	ROM_LOAD16_WORD_SWAP( "vama.10a", 0x380000, 0x80000, 0x5e03d747 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "vam.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.14m",   0x1000000, 0x100000, 0xbd87243c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.16m",   0x1000002, 0x100000, 0xafec855f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.18m",   0x1000004, 0x100000, 0x3a033625 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vam.20m",   0x1000006, 0x100000, 0x2bff6a89 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vam.01",   0x00000, 0x08000, 0x64b685d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vam.02",   0x28000, 0x20000, 0xcf7c97c7 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vam.11m",   0x000000, 0x200000, 0x4a39deb2 )
	ROM_LOAD16_WORD_SWAP( "vam.12m",   0x200000, 0x200000, 0x1a3e5c03 )
ROM_END

ROM_START( ecofghtr )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "uece.03", 0x000000, 0x80000, 0xec2c1137 )
	ROM_LOAD16_WORD_SWAP( "uece.04", 0x080000, 0x80000, 0xb35f99db )
	ROM_LOAD16_WORD_SWAP( "uece.05", 0x100000, 0x80000, 0xd9d42d31 )
	ROM_LOAD16_WORD_SWAP( "uece.06", 0x180000, 0x80000, 0x9d9771cf )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "uec.13m",   0x000000, 0x200000, 0xdcaf1436 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.15m",   0x000002, 0x200000, 0x2807df41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.17m",   0x000004, 0x200000, 0x8a708d02 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.19m",   0x000006, 0x200000, 0xde7be0ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.14m",   0x800000, 0x100000, 0x1a003558 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.16m",   0x800002, 0x100000, 0x4ff8a6f9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.18m",   0x800004, 0x100000, 0xb167ae12 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.20m",   0x800006, 0x100000, 0x1064bdc2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "uec.01",   0x00000, 0x08000, 0xc235bd15 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "uec.11m",   0x000000, 0x200000, 0x81b25d39 )
	ROM_LOAD16_WORD_SWAP( "uec.12m",   0x200000, 0x200000, 0x27729e52 )
ROM_END

ROM_START( ecofghtu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "uecu.03a", 0x000000, 0x80000, 0x22d88a4d )
	ROM_LOAD16_WORD_SWAP( "uecu.04a", 0x080000, 0x80000, 0x6436cfcd )
	ROM_LOAD16_WORD_SWAP( "uecu.05a", 0x100000, 0x80000, 0x336f121b )
	ROM_LOAD16_WORD_SWAP( "uecu.06a", 0x180000, 0x80000, 0x6f99d984 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "uec.13m",   0x000000, 0x200000, 0xdcaf1436 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.15m",   0x000002, 0x200000, 0x2807df41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.17m",   0x000004, 0x200000, 0x8a708d02 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.19m",   0x000006, 0x200000, 0xde7be0ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.14m",   0x800000, 0x100000, 0x1a003558 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.16m",   0x800002, 0x100000, 0x4ff8a6f9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.18m",   0x800004, 0x100000, 0xb167ae12 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.20m",   0x800006, 0x100000, 0x1064bdc2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "uec.01",   0x00000, 0x08000, 0xc235bd15 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "uec.11m",   0x000000, 0x200000, 0x81b25d39 )
	ROM_LOAD16_WORD_SWAP( "uec.12m",   0x200000, 0x200000, 0x27729e52 )
ROM_END

ROM_START( ecofgtu1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "uecu.03", 0x000000, 0x80000, 0x6792480c )
	ROM_LOAD16_WORD_SWAP( "uecu.04", 0x080000, 0x80000, 0x95ce69d5 )
	ROM_LOAD16_WORD_SWAP( "uecu.05", 0x100000, 0x80000, 0x3a1e78ad )
	ROM_LOAD16_WORD_SWAP( "uecu.06", 0x180000, 0x80000, 0xa3e2f3cc )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "uec.13m",   0x000000, 0x200000, 0xdcaf1436 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.15m",   0x000002, 0x200000, 0x2807df41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.17m",   0x000004, 0x200000, 0x8a708d02 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.19m",   0x000006, 0x200000, 0xde7be0ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.14m",   0x800000, 0x100000, 0x1a003558 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.16m",   0x800002, 0x100000, 0x4ff8a6f9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.18m",   0x800004, 0x100000, 0xb167ae12 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.20m",   0x800006, 0x100000, 0x1064bdc2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "uec.01",   0x00000, 0x08000, 0xc235bd15 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "uec.11m",   0x000000, 0x200000, 0x81b25d39 )
	ROM_LOAD16_WORD_SWAP( "uec.12m",   0x200000, 0x200000, 0x27729e52 )
ROM_END

ROM_START( uecology )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "uecj.03", 0x000000, 0x80000, 0x94c40a4c )
	ROM_LOAD16_WORD_SWAP( "uecj.04", 0x080000, 0x80000, 0x8d6e3a09 )
	ROM_LOAD16_WORD_SWAP( "uecj.05", 0x100000, 0x80000, 0x8604ecd7 )
	ROM_LOAD16_WORD_SWAP( "uecj.06", 0x180000, 0x80000, 0xb7e1d31f )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "uec.13m",   0x000000, 0x200000, 0xdcaf1436 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.15m",   0x000002, 0x200000, 0x2807df41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.17m",   0x000004, 0x200000, 0x8a708d02 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.19m",   0x000006, 0x200000, 0xde7be0ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.14m",   0x800000, 0x100000, 0x1a003558 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.16m",   0x800002, 0x100000, 0x4ff8a6f9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.18m",   0x800004, 0x100000, 0xb167ae12 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.20m",   0x800006, 0x100000, 0x1064bdc2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "uec.01",   0x00000, 0x08000, 0xc235bd15 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "uec.11m",   0x000000, 0x200000, 0x81b25d39 )
	ROM_LOAD16_WORD_SWAP( "uec.12m",   0x200000, 0x200000, 0x27729e52 )
ROM_END

ROM_START( ecofghta )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ueca.03", 0x000000, 0x80000, 0xbd4589b1 )
	ROM_LOAD16_WORD_SWAP( "ueca.04", 0x080000, 0x80000, 0x1d134b7d )
	ROM_LOAD16_WORD_SWAP( "ueca.05", 0x100000, 0x80000, 0x9c581fc7 )
	ROM_LOAD16_WORD_SWAP( "ueca.06", 0x180000, 0x80000, 0xc92a7c50 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "uec.13m",   0x000000, 0x200000, 0xdcaf1436 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.15m",   0x000002, 0x200000, 0x2807df41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.17m",   0x000004, 0x200000, 0x8a708d02 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.19m",   0x000006, 0x200000, 0xde7be0ef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.14m",   0x800000, 0x100000, 0x1a003558 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.16m",   0x800002, 0x100000, 0x4ff8a6f9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.18m",   0x800004, 0x100000, 0xb167ae12 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "uec.20m",   0x800006, 0x100000, 0x1064bdc2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "uec.01",   0x00000, 0x08000, 0xc235bd15 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "uec.11m",   0x000000, 0x200000, 0x81b25d39 )
	ROM_LOAD16_WORD_SWAP( "uec.12m",   0x200000, 0x200000, 0x27729e52 )
ROM_END

ROM_START( gigawing )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ggwu.03", 0x000000, 0x80000, 0xac725eb2 )
	ROM_LOAD16_WORD_SWAP( "ggwu.04", 0x080000, 0x80000, 0x392f4118 )
	ROM_LOAD16_WORD_SWAP( "ggwu.05", 0x100000, 0x80000, 0x3239d642 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "ggw.13m",   0x000000, 0x400000, 0x105530a4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ggw.15m",   0x000002, 0x400000, 0x9e774ab9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ggw.17m",   0x000004, 0x400000, 0x466e0ba4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ggw.19m",   0x000006, 0x400000, 0x840c8dea , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ggw.01",   0x00000, 0x08000, 0x4c6351d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "ggw.11m",   0x000000, 0x400000, 0xe172acf5 )
	ROM_LOAD16_WORD_SWAP( "ggw.12m",   0x400000, 0x400000, 0x4bee4e8f )
ROM_END

ROM_START( gwingj )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ggwj.03a", 0x000000, 0x80000, 0xfdd23b91 )
	ROM_LOAD16_WORD_SWAP( "ggwj.04a", 0x080000, 0x80000, 0x8c6e093c )
	ROM_LOAD16_WORD_SWAP( "ggwj.05a", 0x100000, 0x80000, 0x43811454 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "ggw.13m",   0x000000, 0x400000, 0x105530a4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ggw.15m",   0x000002, 0x400000, 0x9e774ab9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ggw.17m",   0x000004, 0x400000, 0x466e0ba4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ggw.19m",   0x000006, 0x400000, 0x840c8dea , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ggw.01",   0x00000, 0x08000, 0x4c6351d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "ggw.11m",   0x000000, 0x400000, 0xe172acf5 )
	ROM_LOAD16_WORD_SWAP( "ggw.12m",   0x400000, 0x400000, 0x4bee4e8f )
ROM_END

ROM_START( gwinga )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ggwa.03a", 0x000000, 0x80000, 0x116f8837 )
	ROM_LOAD16_WORD_SWAP( "ggwa.04a", 0x080000, 0x80000, 0xe6e3f0c4 )
	ROM_LOAD16_WORD_SWAP( "ggwa.05a", 0x100000, 0x80000, 0x465e8ac9 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "ggw.13m",   0x000000, 0x400000, 0x105530a4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ggw.15m",   0x000002, 0x400000, 0x9e774ab9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ggw.17m",   0x000004, 0x400000, 0x466e0ba4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ggw.19m",   0x000006, 0x400000, 0x840c8dea , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ggw.01",   0x00000, 0x08000, 0x4c6351d5 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "ggw.11m",   0x000000, 0x400000, 0xe172acf5 )
	ROM_LOAD16_WORD_SWAP( "ggw.12m",   0x400000, 0x400000, 0x4bee4e8f )
ROM_END


ROM_START( hsf2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "hs2a.03",  0x000000, 0x80000, 0xd50a17e0 )
	ROM_LOAD16_WORD_SWAP( "hs2a.04",  0x080000, 0x80000, 0xa27f42de )
	ROM_LOAD16_WORD_SWAP( "hs2.05",   0x100000, 0x80000, 0xdde34a35 )
	ROM_LOAD16_WORD_SWAP( "hs2.06",   0x180000, 0x80000, 0xf4e56dda )
	ROM_LOAD16_WORD_SWAP( "hs2a.07",  0x200000, 0x80000, 0xee4420fc )
	ROM_LOAD16_WORD_SWAP( "hs2.08",   0x280000, 0x80000, 0xc9441533 )
	ROM_LOAD16_WORD_SWAP( "hs2.09",   0x300000, 0x80000, 0x3fc638a8 )
	ROM_LOAD16_WORD_SWAP( "hs2.10",   0x380000, 0x80000, 0x20d0f9e4 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "hs2.13m",   0x0000000, 0x800000, 0xa6ecab17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "hs2.15m",   0x0000002, 0x800000, 0x10a0ae4d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "hs2.17m",   0x0000004, 0x800000, 0xadfa7726 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "hs2.19m",   0x0000006, 0x800000, 0xbb3ae322 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "hs2.01",   0x00000, 0x08000, 0xc1a13786 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "hs2.02",   0x28000, 0x20000, 0x2d8794aa )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "hs2.11m",   0x000000, 0x800000, 0x0e15c359 )
ROM_END

ROM_START( hsf2j )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "hs2j.03",  0x000000, 0x80000, 0x00738f73 )
	ROM_LOAD16_WORD_SWAP( "hs2j.04",  0x080000, 0x80000, 0x40072c4a )
	ROM_LOAD16_WORD_SWAP( "hs2.05",   0x100000, 0x80000, 0xdde34a35 )
	ROM_LOAD16_WORD_SWAP( "hs2.06",   0x180000, 0x80000, 0xf4e56dda )
	ROM_LOAD16_WORD_SWAP( "hs2j.07",  0x200000, 0x80000, 0x09fe85b4 )
	ROM_LOAD16_WORD_SWAP( "hs2.08",   0x280000, 0x80000, 0xc9441533 )
	ROM_LOAD16_WORD_SWAP( "hs2.09",   0x300000, 0x80000, 0x3fc638a8 )
	ROM_LOAD16_WORD_SWAP( "hs2.10",   0x380000, 0x80000, 0x20d0f9e4 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "hs2.13m",   0x0000000, 0x800000, 0xa6ecab17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "hs2.15m",   0x0000002, 0x800000, 0x10a0ae4d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "hs2.17m",   0x0000004, 0x800000, 0xadfa7726 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "hs2.19m",   0x0000006, 0x800000, 0xbb3ae322 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "hs2.01",   0x00000, 0x08000, 0xc1a13786 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "hs2.02",   0x28000, 0x20000, 0x2d8794aa )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "hs2.11m",   0x000000, 0x800000, 0x0e15c359 )
ROM_END


ROM_START( jyangoku )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "maj_j03.6a", 0x000000, 0x80000, 0x4614a3b2 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "maj1_d.simm1",   0x0000000, 0x200000, 0xba0fe27b , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "maj1_c.simm1",   0x0000001, 0x200000, 0x2cd141bf , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "maj1_b.simm1",   0x0000002, 0x200000, 0xe29e4c26 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "maj1_a.simm1",   0x0000003, 0x200000, 0x7f68b88a , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "maj3_d.simm3",   0x0000004, 0x200000, 0x3aaeb90b , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "maj3_c.simm3",   0x0000005, 0x200000, 0x97894cea , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "maj3_b.simm3",   0x0000006, 0x200000, 0xec737d9d , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "maj3_a.simm3",   0x0000007, 0x200000, 0xc23b6f22 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "maj_01.1a",  0x00000, 0x08000, 0x1fe8c213 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_BYTE( "maj5_a.simm5",   0x000000, 0x200000, 0x5ad9ee53 ) // ROM on a simm
	ROM_LOAD16_BYTE( "maj5_b.simm5",   0x000001, 0x200000, 0xefb3dbfb ) // ROM on a simm
ROM_END

ROM_START( megaman2 )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "rm2u.03", 0x000000, 0x80000, 0x8ffc2cd1 )
	ROM_LOAD16_WORD_SWAP( "rm2u.04", 0x080000, 0x80000, 0xbb30083a )
	ROM_LOAD16_WORD_SWAP( "rm2.05",  0x100000, 0x80000, 0x02ee9efc )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "rm2.14m",   0x800000, 0x200000, 0x9b1f00b4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rm2.16m",   0x800002, 0x200000, 0xc2bb0c24 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rm2.18m",   0x800004, 0x200000, 0x12257251 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rm2.20m",   0x800006, 0x200000, 0xf9b6e786 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "rm2.01a",  0x00000, 0x08000, 0xd18e7859 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "rm2.02",   0x28000, 0x20000, 0xc463ece0 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "rm2.11m",   0x000000, 0x200000, 0x2106174d )
	ROM_LOAD16_WORD_SWAP( "rm2.12m",   0x200000, 0x200000, 0x546c1636 )
ROM_END

ROM_START( megamn2a )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "rm2a.03", 0x000000, 0x80000, 0x2b330ca7 )
	ROM_LOAD16_WORD_SWAP( "rm2a.04", 0x080000, 0x80000, 0x8b47942b )
	ROM_LOAD16_WORD_SWAP( "rm2.05",  0x100000, 0x80000, 0x02ee9efc )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "rm2.14m",   0x800000, 0x200000, 0x9b1f00b4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rm2.16m",   0x800002, 0x200000, 0xc2bb0c24 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rm2.18m",   0x800004, 0x200000, 0x12257251 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rm2.20m",   0x800006, 0x200000, 0xf9b6e786 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "rm2.01a",  0x00000, 0x08000, 0xd18e7859 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "rm2.02",   0x28000, 0x20000, 0xc463ece0 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "rm2.11m",   0x000000, 0x200000, 0x2106174d )
	ROM_LOAD16_WORD_SWAP( "rm2.12m",   0x200000, 0x200000, 0x546c1636 )
ROM_END

ROM_START( rckman2j )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "rm2j.03", 0x000000, 0x80000, 0xdbaa1437 )
	ROM_LOAD16_WORD_SWAP( "rm2j.04", 0x080000, 0x80000, 0xcf5ba612 )
	ROM_LOAD16_WORD_SWAP( "rm2.05",  0x100000, 0x80000, 0x02ee9efc )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "rm2.14m",   0x800000, 0x200000, 0x9b1f00b4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rm2.16m",   0x800002, 0x200000, 0xc2bb0c24 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rm2.18m",   0x800004, 0x200000, 0x12257251 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rm2.20m",   0x800006, 0x200000, 0xf9b6e786 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "rm2.01a",  0x00000, 0x08000, 0xd18e7859 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "rm2.02",   0x28000, 0x20000, 0xc463ece0 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "rm2.11m",   0x000000, 0x200000, 0x2106174d )
	ROM_LOAD16_WORD_SWAP( "rm2.12m",   0x200000, 0x200000, 0x546c1636 )
ROM_END

ROM_START( mmatrix )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mmxu.03", 0x000000, 0x80000, 0xab65b599 )
	ROM_LOAD16_WORD_SWAP( "mmxu.04", 0x080000, 0x80000, 0x0135fc6c )
	ROM_LOAD16_WORD_SWAP( "mmxu.05", 0x100000, 0x80000, 0xf1fd2b84 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mmx.13m",   0x0000000, 0x400000, 0x04748718 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.15m",   0x0000002, 0x400000, 0x38074f44 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.17m",   0x0000004, 0x400000, 0xe4635e35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.19m",   0x0000006, 0x400000, 0x4400a3f2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.14m",   0x1000000, 0x400000, 0xd52bf491 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.16m",   0x1000002, 0x400000, 0x23f70780 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.18m",   0x1000004, 0x400000, 0x2562c9d5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.20m",   0x1000006, 0x400000, 0x583a9687 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mmx.01",   0x00000, 0x08000, 0xc57e8171 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mmx.11m",   0x000000, 0x400000, 0x4180b39f )
	ROM_LOAD16_WORD_SWAP( "mmx.12m",   0x400000, 0x400000, 0x95e22a59 )
ROM_END

ROM_START( mmatrixj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mmxj.03", 0x000000, 0x80000, 0x1d5de213 )
	ROM_LOAD16_WORD_SWAP( "mmxj.04", 0x080000, 0x80000, 0xd943a339 )
	ROM_LOAD16_WORD_SWAP( "mmxj.05", 0x100000, 0x80000, 0x0c8b4abb )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mmx.13m",   0x0000000, 0x400000, 0x04748718 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.15m",   0x0000002, 0x400000, 0x38074f44 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.17m",   0x0000004, 0x400000, 0xe4635e35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.19m",   0x0000006, 0x400000, 0x4400a3f2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.14m",   0x1000000, 0x400000, 0xd52bf491 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.16m",   0x1000002, 0x400000, 0x23f70780 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.18m",   0x1000004, 0x400000, 0x2562c9d5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mmx.20m",   0x1000006, 0x400000, 0x583a9687 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mmx.01",   0x00000, 0x08000, 0xc57e8171 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mmx.11m",   0x000000, 0x400000, 0x4180b39f )
	ROM_LOAD16_WORD_SWAP( "mmx.12m",   0x400000, 0x400000, 0x95e22a59 )
ROM_END

ROM_START( msh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mshe.03e", 0x000000, 0x80000, 0xbd951414 )
	ROM_LOAD16_WORD_SWAP( "mshe.04e", 0x080000, 0x80000, 0x19dd42f2 )
	ROM_LOAD16_WORD_SWAP( "msh.05",   0x100000, 0x80000, 0x6a091b9e )
	ROM_LOAD16_WORD_SWAP( "msh.06b",  0x180000, 0x80000, 0x803e3fa4 )
	ROM_LOAD16_WORD_SWAP( "msh.07a",  0x200000, 0x80000, 0xc45f8e27 )
	ROM_LOAD16_WORD_SWAP( "msh.08a",  0x280000, 0x80000, 0x9ca6f12c )
	ROM_LOAD16_WORD_SWAP( "msh.09a",  0x300000, 0x80000, 0x82ec27af )
	ROM_LOAD16_WORD_SWAP( "msh.10b",  0x380000, 0x80000, 0x8d931196 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "msh.13m",   0x0000000, 0x400000, 0x09d14566 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.15m",   0x0000002, 0x400000, 0xee962057 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.17m",   0x0000004, 0x400000, 0x604ece14 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.19m",   0x0000006, 0x400000, 0x94a731e8 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.14m",   0x1000000, 0x400000, 0x4197973e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.16m",   0x1000002, 0x400000, 0x438da4a0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.18m",   0x1000004, 0x400000, 0x4db92d94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.20m",   0x1000006, 0x400000, 0xa2b0c6c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "msh.01",   0x00000, 0x08000, 0xc976e6f9 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "msh.02",   0x28000, 0x20000, 0xce67d0d9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "msh.11m",   0x000000, 0x200000, 0x37ac6d30 )
	ROM_LOAD16_WORD_SWAP( "msh.12m",   0x200000, 0x200000, 0xde092570 )
ROM_END

ROM_START( mshu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mshu.03", 0x000000, 0x80000, 0xd2805bdd )
	ROM_LOAD16_WORD_SWAP( "mshu.04", 0x080000, 0x80000, 0x743f96ff )
	ROM_LOAD16_WORD_SWAP( "msh.05",  0x100000, 0x80000, 0x6a091b9e )
	ROM_LOAD16_WORD_SWAP( "msh.06b", 0x180000, 0x80000, 0x803e3fa4 )
	ROM_LOAD16_WORD_SWAP( "msh.07a", 0x200000, 0x80000, 0xc45f8e27 )
	ROM_LOAD16_WORD_SWAP( "msh.08a", 0x280000, 0x80000, 0x9ca6f12c )
	ROM_LOAD16_WORD_SWAP( "msh.09a", 0x300000, 0x80000, 0x82ec27af )
	ROM_LOAD16_WORD_SWAP( "msh.10b", 0x380000, 0x80000, 0x8d931196 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "msh.13m",   0x0000000, 0x400000, 0x09d14566 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.15m",   0x0000002, 0x400000, 0xee962057 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.17m",   0x0000004, 0x400000, 0x604ece14 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.19m",   0x0000006, 0x400000, 0x94a731e8 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.14m",   0x1000000, 0x400000, 0x4197973e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.16m",   0x1000002, 0x400000, 0x438da4a0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.18m",   0x1000004, 0x400000, 0x4db92d94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.20m",   0x1000006, 0x400000, 0xa2b0c6c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "msh.01",   0x00000, 0x08000, 0xc976e6f9 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "msh.02",   0x28000, 0x20000, 0xce67d0d9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "msh.11m",   0x000000, 0x200000, 0x37ac6d30 )
	ROM_LOAD16_WORD_SWAP( "msh.12m",   0x200000, 0x200000, 0xde092570 )
ROM_END

ROM_START( mshj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mshj.03g", 0x000000, 0x80000, 0x261f4091 )
	ROM_LOAD16_WORD_SWAP( "mshj.04g", 0x080000, 0x80000, 0x61d791c6 )
	ROM_LOAD16_WORD_SWAP( "msh.05a",  0x100000, 0x80000, 0xf37539e6 )
	ROM_LOAD16_WORD_SWAP( "msh.06b",  0x180000, 0x80000, 0x803e3fa4 )
	ROM_LOAD16_WORD_SWAP( "msh.07a",  0x200000, 0x80000, 0xc45f8e27 )
	ROM_LOAD16_WORD_SWAP( "msh.08a",  0x280000, 0x80000, 0x9ca6f12c )
	ROM_LOAD16_WORD_SWAP( "msh.09a",  0x300000, 0x80000, 0x82ec27af )
	ROM_LOAD16_WORD_SWAP( "msh.10b",  0x380000, 0x80000, 0x8d931196 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "msh.13m",   0x0000000, 0x400000, 0x09d14566 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.15m",   0x0000002, 0x400000, 0xee962057 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.17m",   0x0000004, 0x400000, 0x604ece14 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.19m",   0x0000006, 0x400000, 0x94a731e8 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.14m",   0x1000000, 0x400000, 0x4197973e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.16m",   0x1000002, 0x400000, 0x438da4a0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.18m",   0x1000004, 0x400000, 0x4db92d94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.20m",   0x1000006, 0x400000, 0xa2b0c6c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "msh.01",   0x00000, 0x08000, 0xc976e6f9 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "msh.02",   0x28000, 0x20000, 0xce67d0d9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "msh.11m",   0x000000, 0x200000, 0x37ac6d30 )
	ROM_LOAD16_WORD_SWAP( "msh.12m",   0x200000, 0x200000, 0xde092570 )
ROM_END

ROM_START( mshjr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mshj.03f", 0x000000, 0x80000, 0xff172fd2 )
	ROM_LOAD16_WORD_SWAP( "mshj.04f", 0x080000, 0x80000, 0xebbb205a )
	ROM_LOAD16_WORD_SWAP( "msh.05",   0x100000, 0x80000, 0x6a091b9e )
	ROM_LOAD16_WORD_SWAP( "msh.06b",  0x180000, 0x80000, 0x803e3fa4 )
	ROM_LOAD16_WORD_SWAP( "msh.07a",  0x200000, 0x80000, 0xc45f8e27 )
	ROM_LOAD16_WORD_SWAP( "msh.08a",  0x280000, 0x80000, 0x9ca6f12c )
	ROM_LOAD16_WORD_SWAP( "msh.09a",  0x300000, 0x80000, 0x82ec27af )
	ROM_LOAD16_WORD_SWAP( "msh.10b",  0x380000, 0x80000, 0x8d931196 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "msh.13m",   0x0000000, 0x400000, 0x09d14566 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.15m",   0x0000002, 0x400000, 0xee962057 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.17m",   0x0000004, 0x400000, 0x604ece14 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.19m",   0x0000006, 0x400000, 0x94a731e8 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.14m",   0x1000000, 0x400000, 0x4197973e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.16m",   0x1000002, 0x400000, 0x438da4a0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.18m",   0x1000004, 0x400000, 0x4db92d94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.20m",   0x1000006, 0x400000, 0xa2b0c6c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "msh.01",   0x00000, 0x08000, 0xc976e6f9 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "msh.02",   0x28000, 0x20000, 0xce67d0d9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "msh.11m",   0x000000, 0x200000, 0x37ac6d30 )
	ROM_LOAD16_WORD_SWAP( "msh.12m",   0x200000, 0x200000, 0xde092570 )
ROM_END

ROM_START( msha )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "msha.03e", 0x000000, 0x80000, 0xec84ec44 )
	ROM_LOAD16_WORD_SWAP( "msha.04e", 0x080000, 0x80000, 0x098b8503 )
	ROM_LOAD16_WORD_SWAP( "msh.05",   0x100000, 0x80000, 0x6a091b9e )
	ROM_LOAD16_WORD_SWAP( "msh.06b",  0x180000, 0x80000, 0x803e3fa4 )
	ROM_LOAD16_WORD_SWAP( "msh.07a",  0x200000, 0x80000, 0xc45f8e27 )
	ROM_LOAD16_WORD_SWAP( "msh.08a",  0x280000, 0x80000, 0x9ca6f12c )
	ROM_LOAD16_WORD_SWAP( "msh.09a",  0x300000, 0x80000, 0x82ec27af )
	ROM_LOAD16_WORD_SWAP( "msh.10b",  0x380000, 0x80000, 0x8d931196 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "msh.13m",   0x0000000, 0x400000, 0x09d14566 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.15m",   0x0000002, 0x400000, 0xee962057 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.17m",   0x0000004, 0x400000, 0x604ece14 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.19m",   0x0000006, 0x400000, 0x94a731e8 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.14m",   0x1000000, 0x400000, 0x4197973e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.16m",   0x1000002, 0x400000, 0x438da4a0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.18m",   0x1000004, 0x400000, 0x4db92d94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.20m",   0x1000006, 0x400000, 0xa2b0c6c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "msh.01",   0x00000, 0x08000, 0xc976e6f9 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "msh.02",   0x28000, 0x20000, 0xce67d0d9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "msh.11m",   0x000000, 0x200000, 0x37ac6d30 )
	ROM_LOAD16_WORD_SWAP( "msh.12m",   0x200000, 0x200000, 0xde092570 )
ROM_END

ROM_START( mshh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mshh.03c", 0x000000, 0x80000, 0x8d84b0fa )
	ROM_LOAD16_WORD_SWAP( "mshh.04c", 0x080000, 0x80000, 0xd638f601 )
	ROM_LOAD16_WORD_SWAP( "msh.05a",  0x100000, 0x80000, 0xf37539e6 )
	ROM_LOAD16_WORD_SWAP( "msh.06b",  0x180000, 0x80000, 0x803e3fa4 )
	ROM_LOAD16_WORD_SWAP( "msh.07a",  0x200000, 0x80000, 0xc45f8e27 )
	ROM_LOAD16_WORD_SWAP( "msh.08a",  0x280000, 0x80000, 0x9ca6f12c )
	ROM_LOAD16_WORD_SWAP( "msh.09a",  0x300000, 0x80000, 0x82ec27af )
	ROM_LOAD16_WORD_SWAP( "msh.10b",  0x380000, 0x80000, 0x8d931196 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "msh.13m",   0x0000000, 0x400000, 0x09d14566 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.15m",   0x0000002, 0x400000, 0xee962057 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.17m",   0x0000004, 0x400000, 0x604ece14 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.19m",   0x0000006, 0x400000, 0x94a731e8 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.14m",   0x1000000, 0x400000, 0x4197973e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.16m",   0x1000002, 0x400000, 0x438da4a0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.18m",   0x1000004, 0x400000, 0x4db92d94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.20m",   0x1000006, 0x400000, 0xa2b0c6c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "msh.01",   0x00000, 0x08000, 0xc976e6f9 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "msh.02",   0x28000, 0x20000, 0xce67d0d9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "msh.11m",   0x000000, 0x200000, 0x37ac6d30 )
	ROM_LOAD16_WORD_SWAP( "msh.12m",   0x200000, 0x200000, 0xde092570 )
ROM_END

ROM_START( mshb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mshb.03c", 0x000000, 0x80000, 0x19697f74 )
	ROM_LOAD16_WORD_SWAP( "mshb.04c", 0x080000, 0x80000, 0x95317a6f )
	ROM_LOAD16_WORD_SWAP( "msh.05a",  0x100000, 0x80000, 0xf37539e6 )
	ROM_LOAD16_WORD_SWAP( "msh.06b",  0x180000, 0x80000, 0x803e3fa4 )
	ROM_LOAD16_WORD_SWAP( "msh.07a",  0x200000, 0x80000, 0xc45f8e27 )
	ROM_LOAD16_WORD_SWAP( "msh.08a",  0x280000, 0x80000, 0x9ca6f12c )
	ROM_LOAD16_WORD_SWAP( "msh.09a",  0x300000, 0x80000, 0x82ec27af )
	ROM_LOAD16_WORD_SWAP( "msh.10b",  0x380000, 0x80000, 0x8d931196 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "msh.13m",   0x0000000, 0x400000, 0x09d14566 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.15m",   0x0000002, 0x400000, 0xee962057 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.17m",   0x0000004, 0x400000, 0x604ece14 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.19m",   0x0000006, 0x400000, 0x94a731e8 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.14m",   0x1000000, 0x400000, 0x4197973e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.16m",   0x1000002, 0x400000, 0x438da4a0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.18m",   0x1000004, 0x400000, 0x4db92d94 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "msh.20m",   0x1000006, 0x400000, 0xa2b0c6c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "msh.01",   0x00000, 0x08000, 0xc976e6f9 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "msh.02",   0x28000, 0x20000, 0xce67d0d9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "msh.11m",   0x000000, 0x200000, 0x37ac6d30 )
	ROM_LOAD16_WORD_SWAP( "msh.12m",   0x200000, 0x200000, 0xde092570 )
ROM_END

ROM_START( mshvsf )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvse.03f", 0x000000, 0x80000, 0xb72dc199 )
	ROM_LOAD16_WORD_SWAP( "mvse.04f", 0x080000, 0x80000, 0x6ef799f9 )
	ROM_LOAD16_WORD_SWAP( "mvs.05a",  0x100000, 0x80000, 0x1a5de0cb )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsu.03g", 0x000000, 0x80000, 0x0664ab15 )
	ROM_LOAD16_WORD_SWAP( "mvsu.04g", 0x080000, 0x80000, 0x97e060ee )
	ROM_LOAD16_WORD_SWAP( "mvs.05d",  0x100000, 0x80000, 0x921fc542 )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfu1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsu.03d", 0x000000, 0x80000, 0xae60a66a )
	ROM_LOAD16_WORD_SWAP( "mvsu.04d", 0x080000, 0x80000, 0x91f67d8a )
	ROM_LOAD16_WORD_SWAP( "mvs.05a",  0x100000, 0x80000, 0x1a5de0cb )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsj.03i", 0x000000, 0x80000, 0xd8cbb691 )
	ROM_LOAD16_WORD_SWAP( "mvsj.04i", 0x080000, 0x80000, 0x32741ace )
	ROM_LOAD16_WORD_SWAP( "mvs.05h",  0x100000, 0x80000, 0x77870dc3 )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfj1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsj.03h", 0x000000, 0x80000, 0xfbe2115f )
	ROM_LOAD16_WORD_SWAP( "mvsj.04h", 0x080000, 0x80000, 0xb528a367 )
	ROM_LOAD16_WORD_SWAP( "mvs.05g",  0x100000, 0x80000, 0x9515a245 )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfj2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsj.03g", 0x000000, 0x80000, 0xfdfa7e26 )
	ROM_LOAD16_WORD_SWAP( "mvsj.04g", 0x080000, 0x80000, 0xc921825f )
	ROM_LOAD16_WORD_SWAP( "mvs.05a",  0x100000, 0x80000, 0x1a5de0cb )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsh.03f", 0x000000, 0x80000, 0x4f60f41e )
	ROM_LOAD16_WORD_SWAP( "mvsh.04f", 0x080000, 0x80000, 0xdc08ec12 )
	ROM_LOAD16_WORD_SWAP( "mvs.05a",  0x100000, 0x80000, 0x1a5de0cb )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfa )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsa.03f", 0x000000, 0x80000, 0x5b863716 )
	ROM_LOAD16_WORD_SWAP( "mvsa.04f", 0x080000, 0x80000, 0x4886e65f )
	ROM_LOAD16_WORD_SWAP( "mvs.05a",  0x100000, 0x80000, 0x1a5de0cb )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfa1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsa.03", 0x000000, 0x80000, 0x92ef1933 )
	ROM_LOAD16_WORD_SWAP( "mvsa.04", 0x080000, 0x80000, 0x4b24373c )
	ROM_LOAD16_WORD_SWAP( "mvs.05",  0x100000, 0x80000, 0xac180c1c )
	ROM_LOAD16_WORD_SWAP( "mvs.06a", 0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b", 0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a", 0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b", 0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b", 0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsb.03g", 0x000000, 0x80000, 0x143895ef )
	ROM_LOAD16_WORD_SWAP( "mvsb.04g", 0x080000, 0x80000, 0xdd8a886c )
	ROM_LOAD16_WORD_SWAP( "mvs.05d",  0x100000, 0x80000, 0x921fc542 )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mshvsfb1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvsb.03f", 0x000000, 0x80000, 0x9c4bb950 )
	ROM_LOAD16_WORD_SWAP( "mvsb.04f", 0x080000, 0x80000, 0xd3320d13 )
	ROM_LOAD16_WORD_SWAP( "mvs.05a",  0x100000, 0x80000, 0x1a5de0cb )
	ROM_LOAD16_WORD_SWAP( "mvs.06a",  0x180000, 0x80000, 0x959f3030 )
	ROM_LOAD16_WORD_SWAP( "mvs.07b",  0x200000, 0x80000, 0x7f915bdb )
	ROM_LOAD16_WORD_SWAP( "mvs.08a",  0x280000, 0x80000, 0xc2813884 )
	ROM_LOAD16_WORD_SWAP( "mvs.09b",  0x300000, 0x80000, 0x3ba08818 )
	ROM_LOAD16_WORD_SWAP( "mvs.10b",  0x380000, 0x80000, 0xcf0dba98 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvs.13m",   0x0000000, 0x400000, 0x29b05fd9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.15m",   0x0000002, 0x400000, 0xfaddccf1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.17m",   0x0000004, 0x400000, 0x97aaf4c7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.19m",   0x0000006, 0x400000, 0xcb70e915 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.14m",   0x1000000, 0x400000, 0xb3b1972d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.16m",   0x1000002, 0x400000, 0x08aadb5d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.18m",   0x1000004, 0x400000, 0xc1228b35 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvs.20m",   0x1000006, 0x400000, 0x366cc6c2 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvs.01",   0x00000, 0x08000, 0x68252324 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvs.02",   0x28000, 0x20000, 0xb34e773d )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvs.11m",   0x000000, 0x400000, 0x86219770 )
	ROM_LOAD16_WORD_SWAP( "mvs.12m",   0x400000, 0x400000, 0xf2fd7f68 )
ROM_END

ROM_START( mvsc )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvce.03", 0x000000, 0x80000, 0xe0633fc0 )
	ROM_LOAD16_WORD_SWAP( "mvce.04", 0x080000, 0x80000, 0xa450a251 )
	ROM_LOAD16_WORD_SWAP( "mvc.05",  0x100000, 0x80000, 0x7db71ce9 )
	ROM_LOAD16_WORD_SWAP( "mvc.06",  0x180000, 0x80000, 0x4b0b6d3e )
	ROM_LOAD16_WORD_SWAP( "mvc.07",  0x200000, 0x80000, 0xc3baa32b )
	ROM_LOAD16_WORD_SWAP( "mvc.08",  0x280000, 0x80000, 0xbc002fcd )
	ROM_LOAD16_WORD_SWAP( "mvc.09",  0x300000, 0x80000, 0xc67b26df )
	ROM_LOAD16_WORD_SWAP( "mvc.10",  0x380000, 0x80000, 0x0fdd1e26 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvc.13m",   0x0000000, 0x400000, 0xfa5f74bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.15m",   0x0000002, 0x400000, 0x71938a8f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.17m",   0x0000004, 0x400000, 0x92741d07 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.19m",   0x0000006, 0x400000, 0xbcb72fc6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.14m",   0x1000000, 0x400000, 0x7f1df4e4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.16m",   0x1000002, 0x400000, 0x90bd3203 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.18m",   0x1000004, 0x400000, 0x67aaf727 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.20m",   0x1000006, 0x400000, 0x8b0bade8 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvc.01",   0x00000, 0x08000, 0x41629e95 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvc.02",   0x28000, 0x20000, 0x963abf6b )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvc.11m",   0x000000, 0x400000, 0x850fe663 )
	ROM_LOAD16_WORD_SWAP( "mvc.12m",   0x400000, 0x400000, 0x7ccb1896 )
ROM_END

ROM_START( mvscu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvcu.03d", 0x000000, 0x80000, 0xc6007557 )
	ROM_LOAD16_WORD_SWAP( "mvcu.04d", 0x080000, 0x80000, 0x724b2b20 )
	ROM_LOAD16_WORD_SWAP( "mvc.05a",  0x100000, 0x80000, 0x2d8c8e86 )
	ROM_LOAD16_WORD_SWAP( "mvc.06a",  0x180000, 0x80000, 0x8528e1f5 )
	ROM_LOAD16_WORD_SWAP( "mvc.07",   0x200000, 0x80000, 0xc3baa32b )
	ROM_LOAD16_WORD_SWAP( "mvc.08",   0x280000, 0x80000, 0xbc002fcd )
	ROM_LOAD16_WORD_SWAP( "mvc.09",   0x300000, 0x80000, 0xc67b26df )
	ROM_LOAD16_WORD_SWAP( "mvc.10",   0x380000, 0x80000, 0x0fdd1e26 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvc.13m",   0x0000000, 0x400000, 0xfa5f74bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.15m",   0x0000002, 0x400000, 0x71938a8f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.17m",   0x0000004, 0x400000, 0x92741d07 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.19m",   0x0000006, 0x400000, 0xbcb72fc6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.14m",   0x1000000, 0x400000, 0x7f1df4e4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.16m",   0x1000002, 0x400000, 0x90bd3203 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.18m",   0x1000004, 0x400000, 0x67aaf727 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.20m",   0x1000006, 0x400000, 0x8b0bade8 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvc.01",   0x00000, 0x08000, 0x41629e95 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvc.02",   0x28000, 0x20000, 0x963abf6b )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvc.11m",   0x000000, 0x400000, 0x850fe663 )
	ROM_LOAD16_WORD_SWAP( "mvc.12m",   0x400000, 0x400000, 0x7ccb1896 )
ROM_END

ROM_START( mvscj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvcj.03a", 0x000000, 0x80000, 0x3df18879 )
	ROM_LOAD16_WORD_SWAP( "mvcj.04a", 0x080000, 0x80000, 0x07d212e8 )
	ROM_LOAD16_WORD_SWAP( "mvc.05a",  0x100000, 0x80000, 0x2d8c8e86 )
	ROM_LOAD16_WORD_SWAP( "mvc.06a",  0x180000, 0x80000, 0x8528e1f5 )
	ROM_LOAD16_WORD_SWAP( "mvc.07",   0x200000, 0x80000, 0xc3baa32b )
	ROM_LOAD16_WORD_SWAP( "mvc.08",   0x280000, 0x80000, 0xbc002fcd )
	ROM_LOAD16_WORD_SWAP( "mvc.09",   0x300000, 0x80000, 0xc67b26df )
	ROM_LOAD16_WORD_SWAP( "mvc.10",   0x380000, 0x80000, 0x0fdd1e26 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvc.13m",   0x0000000, 0x400000, 0xfa5f74bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.15m",   0x0000002, 0x400000, 0x71938a8f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.17m",   0x0000004, 0x400000, 0x92741d07 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.19m",   0x0000006, 0x400000, 0xbcb72fc6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.14m",   0x1000000, 0x400000, 0x7f1df4e4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.16m",   0x1000002, 0x400000, 0x90bd3203 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.18m",   0x1000004, 0x400000, 0x67aaf727 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.20m",   0x1000006, 0x400000, 0x8b0bade8 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvc.01",   0x00000, 0x08000, 0x41629e95 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvc.02",   0x28000, 0x20000, 0x963abf6b )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvc.11m",   0x000000, 0x400000, 0x850fe663 )
	ROM_LOAD16_WORD_SWAP( "mvc.12m",   0x400000, 0x400000, 0x7ccb1896 )
ROM_END

ROM_START( mvscjr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvcj.03", 0x000000, 0x80000, 0x2164213f )
	ROM_LOAD16_WORD_SWAP( "mvcj.04", 0x080000, 0x80000, 0xc905c86f )
	ROM_LOAD16_WORD_SWAP( "mvc.05",  0x100000, 0x80000, 0x7db71ce9 )
	ROM_LOAD16_WORD_SWAP( "mvc.06",  0x180000, 0x80000, 0x4b0b6d3e )
	ROM_LOAD16_WORD_SWAP( "mvc.07",  0x200000, 0x80000, 0xc3baa32b )
	ROM_LOAD16_WORD_SWAP( "mvc.08",  0x280000, 0x80000, 0xbc002fcd )
	ROM_LOAD16_WORD_SWAP( "mvc.09",  0x300000, 0x80000, 0xc67b26df )
	ROM_LOAD16_WORD_SWAP( "mvc.10",  0x380000, 0x80000, 0x0fdd1e26 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvc.13m",   0x0000000, 0x400000, 0xfa5f74bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.15m",   0x0000002, 0x400000, 0x71938a8f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.17m",   0x0000004, 0x400000, 0x92741d07 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.19m",   0x0000006, 0x400000, 0xbcb72fc6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.14m",   0x1000000, 0x400000, 0x7f1df4e4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.16m",   0x1000002, 0x400000, 0x90bd3203 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.18m",   0x1000004, 0x400000, 0x67aaf727 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.20m",   0x1000006, 0x400000, 0x8b0bade8 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvc.01",   0x00000, 0x08000, 0x41629e95 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvc.02",   0x28000, 0x20000, 0x963abf6b )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvc.11m",   0x000000, 0x400000, 0x850fe663 )
	ROM_LOAD16_WORD_SWAP( "mvc.12m",   0x400000, 0x400000, 0x7ccb1896 )
ROM_END

ROM_START( mvsca )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvca.03a", 0x000000, 0x80000, 0x2ff4ae25 )
	ROM_LOAD16_WORD_SWAP( "mvca.04a", 0x080000, 0x80000, 0xf28427ef )
	ROM_LOAD16_WORD_SWAP( "mvc.05a",  0x100000, 0x80000, 0x2d8c8e86 )
	ROM_LOAD16_WORD_SWAP( "mvc.06a",  0x180000, 0x80000, 0x8528e1f5 )
	ROM_LOAD16_WORD_SWAP( "mvc.07",   0x200000, 0x80000, 0xc3baa32b )
	ROM_LOAD16_WORD_SWAP( "mvc.08",   0x280000, 0x80000, 0xbc002fcd )
	ROM_LOAD16_WORD_SWAP( "mvc.09",   0x300000, 0x80000, 0xc67b26df )
	ROM_LOAD16_WORD_SWAP( "mvc.10",   0x380000, 0x80000, 0x0fdd1e26 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvc.13m",   0x0000000, 0x400000, 0xfa5f74bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.15m",   0x0000002, 0x400000, 0x71938a8f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.17m",   0x0000004, 0x400000, 0x92741d07 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.19m",   0x0000006, 0x400000, 0xbcb72fc6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.14m",   0x1000000, 0x400000, 0x7f1df4e4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.16m",   0x1000002, 0x400000, 0x90bd3203 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.18m",   0x1000004, 0x400000, 0x67aaf727 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.20m",   0x1000006, 0x400000, 0x8b0bade8 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvc.01",   0x00000, 0x08000, 0x41629e95 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvc.02",   0x28000, 0x20000, 0x963abf6b )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvc.11m",   0x000000, 0x400000, 0x850fe663 )
	ROM_LOAD16_WORD_SWAP( "mvc.12m",   0x400000, 0x400000, 0x7ccb1896 )
ROM_END

ROM_START( mvscar1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvca.03", 0x000000, 0x80000, 0xfe5fa7b9 )
	ROM_LOAD16_WORD_SWAP( "mvca.04", 0x080000, 0x80000, 0x082b701c )
	ROM_LOAD16_WORD_SWAP( "mvc.05",  0x100000, 0x80000, 0x7db71ce9 )
	ROM_LOAD16_WORD_SWAP( "mvc.06",  0x180000, 0x80000, 0x4b0b6d3e )
	ROM_LOAD16_WORD_SWAP( "mvc.07",  0x200000, 0x80000, 0xc3baa32b )
	ROM_LOAD16_WORD_SWAP( "mvc.08",  0x280000, 0x80000, 0xbc002fcd )
	ROM_LOAD16_WORD_SWAP( "mvc.09",  0x300000, 0x80000, 0xc67b26df )
	ROM_LOAD16_WORD_SWAP( "mvc.10",  0x380000, 0x80000, 0x0fdd1e26 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvc.13m",   0x0000000, 0x400000, 0xfa5f74bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.15m",   0x0000002, 0x400000, 0x71938a8f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.17m",   0x0000004, 0x400000, 0x92741d07 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.19m",   0x0000006, 0x400000, 0xbcb72fc6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.14m",   0x1000000, 0x400000, 0x7f1df4e4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.16m",   0x1000002, 0x400000, 0x90bd3203 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.18m",   0x1000004, 0x400000, 0x67aaf727 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.20m",   0x1000006, 0x400000, 0x8b0bade8 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvc.01",   0x00000, 0x08000, 0x41629e95 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvc.02",   0x28000, 0x20000, 0x963abf6b )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvc.11m",   0x000000, 0x400000, 0x850fe663 )
	ROM_LOAD16_WORD_SWAP( "mvc.12m",   0x400000, 0x400000, 0x7ccb1896 )
ROM_END

ROM_START( mvsch )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvch.03", 0x000000, 0x80000, 0x6a0ec9f7 )
	ROM_LOAD16_WORD_SWAP( "mvch.04", 0x080000, 0x80000, 0x00f03fa4 )
	ROM_LOAD16_WORD_SWAP( "mvc.05a", 0x100000, 0x80000, 0x2d8c8e86 )
	ROM_LOAD16_WORD_SWAP( "mvc.06a", 0x180000, 0x80000, 0x8528e1f5 )
	ROM_LOAD16_WORD_SWAP( "mvc.07",  0x200000, 0x80000, 0xc3baa32b )
	ROM_LOAD16_WORD_SWAP( "mvc.08",  0x280000, 0x80000, 0xbc002fcd )
	ROM_LOAD16_WORD_SWAP( "mvc.09",  0x300000, 0x80000, 0xc67b26df )
	ROM_LOAD16_WORD_SWAP( "mvc.10",  0x380000, 0x80000, 0x0fdd1e26 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvc.13m",   0x0000000, 0x400000, 0xfa5f74bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.15m",   0x0000002, 0x400000, 0x71938a8f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.17m",   0x0000004, 0x400000, 0x92741d07 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.19m",   0x0000006, 0x400000, 0xbcb72fc6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.14m",   0x1000000, 0x400000, 0x7f1df4e4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.16m",   0x1000002, 0x400000, 0x90bd3203 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.18m",   0x1000004, 0x400000, 0x67aaf727 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.20m",   0x1000006, 0x400000, 0x8b0bade8 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvc.01",   0x00000, 0x08000, 0x41629e95 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvc.02",   0x28000, 0x20000, 0x963abf6b )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvc.11m",   0x000000, 0x400000, 0x850fe663 )
	ROM_LOAD16_WORD_SWAP( "mvc.12m",   0x400000, 0x400000, 0x7ccb1896 )
ROM_END

ROM_START( mvscb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mvcb.03a", 0x000000, 0x80000, 0x7155953b )
	ROM_LOAD16_WORD_SWAP( "mvcb.04a", 0x080000, 0x80000, 0xfb117d0e )
	ROM_LOAD16_WORD_SWAP( "mvc.05a",  0x100000, 0x80000, 0x2d8c8e86 )
	ROM_LOAD16_WORD_SWAP( "mvc.06a",  0x180000, 0x80000, 0x8528e1f5 )
	ROM_LOAD16_WORD_SWAP( "mvc.07",   0x200000, 0x80000, 0xc3baa32b )
	ROM_LOAD16_WORD_SWAP( "mvc.08",   0x280000, 0x80000, 0xbc002fcd )
	ROM_LOAD16_WORD_SWAP( "mvc.09",   0x300000, 0x80000, 0xc67b26df )
	ROM_LOAD16_WORD_SWAP( "mvc.10",   0x380000, 0x80000, 0x0fdd1e26 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mvc.13m",   0x0000000, 0x400000, 0xfa5f74bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.15m",   0x0000002, 0x400000, 0x71938a8f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.17m",   0x0000004, 0x400000, 0x92741d07 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.19m",   0x0000006, 0x400000, 0xbcb72fc6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.14m",   0x1000000, 0x400000, 0x7f1df4e4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.16m",   0x1000002, 0x400000, 0x90bd3203 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.18m",   0x1000004, 0x400000, 0x67aaf727 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mvc.20m",   0x1000006, 0x400000, 0x8b0bade8 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mvc.01",   0x00000, 0x08000, 0x41629e95 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "mvc.02",   0x28000, 0x20000, 0x963abf6b )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mvc.11m",   0x000000, 0x400000, 0x850fe663 )
	ROM_LOAD16_WORD_SWAP( "mvc.12m",   0x400000, 0x400000, 0x7ccb1896 )
ROM_END

ROM_START( mpang )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mpn.03", 0x000000, 0x80000, 0x6e7ed03c ) /* USA version, but has no "u" in label code */
	ROM_LOAD16_WORD_SWAP( "mpn.04", 0x080000, 0x80000, 0xde079131 )

	ROM_REGION( 0x0800000, REGION_GFX1, 0 )
	ROMX_LOAD( "mpn.13m",   0x0000000, 0x200000, 0xc5f123dc , ROM_GROUPWORD | ROM_SKIP(6) ) /* This version uses roms chips */
	ROMX_LOAD( "mpn.15m",   0x0000002, 0x200000, 0x8e033265 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mpn.17m",   0x0000004, 0x200000, 0xcfcd73d2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "mpn.19m",   0x0000006, 0x200000, 0x2db1ffbc , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mpn.01",   0x00000, 0x08000, 0x90c7adb6 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mpn.q1",   0x000000, 0x100000, 0xd21c1f5a ) /* This version uses roms chips */
	ROM_LOAD16_WORD_SWAP( "mpn.q2",   0x100000, 0x100000, 0xd22090b1 )
	ROM_LOAD16_WORD_SWAP( "mpn.q3",   0x200000, 0x100000, 0x60aa5ef2 )
	ROM_LOAD16_WORD_SWAP( "mpn.q4",   0x300000, 0x100000, 0x3a67d203 )
ROM_END

ROM_START( mpangj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "mpnj.03a", 0x000000, 0x80000, 0xbf597b1c )
	ROM_LOAD16_WORD_SWAP( "mpnj.04a", 0x080000, 0x80000, 0xf4a3ab0f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "mpn-simm.01c",   0x0000000, 0x200000, 0x388db66b , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "mpn-simm.01d",   0x0000001, 0x200000, 0xaff1b494 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "mpn-simm.01a",   0x0000002, 0x200000, 0xa9c4857b , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "mpn-simm.01b",   0x0000003, 0x200000, 0xf759df22 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "mpn-simm.03c",   0x0000004, 0x200000, 0xdec6b720 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "mpn-simm.03d",   0x0000005, 0x200000, 0xf8774c18 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "mpn-simm.03a",   0x0000006, 0x200000, 0xc2aea4ec , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "mpn-simm.03b",   0x0000007, 0x200000, 0x84d6dc33 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "mpn.01",   0x00000, 0x08000, 0x90c7adb6 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "mpn-simm.05a",   0x000000, 0x200000, 0x318a2e21 ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "mpn-simm.05b",   0x200000, 0x200000, 0x5462f4e8 ) // ROM on a simm
ROM_END

ROM_START( nwarr )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vphe.03f", 0x000000, 0x80000, 0xa922c44f )
	ROM_LOAD16_WORD_SWAP( "vphe.04c", 0x080000, 0x80000, 0x7312d890 )
	ROM_LOAD16_WORD_SWAP( "vphe.05d", 0x100000, 0x80000, 0xcde8b506 )
	ROM_LOAD16_WORD_SWAP( "vphe.06c", 0x180000, 0x80000, 0xbe99e7d0 )
	ROM_LOAD16_WORD_SWAP( "vphe.07b", 0x200000, 0x80000, 0x69e0e60c )
	ROM_LOAD16_WORD_SWAP( "vphe.08b", 0x280000, 0x80000, 0xd95a3849 )
	ROM_LOAD16_WORD_SWAP( "vphe.09b", 0x300000, 0x80000, 0x9882561c )
	ROM_LOAD16_WORD_SWAP( "vphe.10b", 0x380000, 0x80000, 0x976fa62f )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vph.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.14m",   0x1000000, 0x400000, 0x7a0e1add , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.16m",   0x1000002, 0x400000, 0x2f41ca75 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.18m",   0x1000004, 0x400000, 0x64498eed , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.20m",   0x1000006, 0x400000, 0x17f2433f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vph.01",   0x00000, 0x08000, 0x5045dcac )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vph.02",   0x28000, 0x20000, 0x86b60e59 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vph.11m",   0x000000, 0x200000, 0xe1837d33 )
	ROM_LOAD16_WORD_SWAP( "vph.12m",   0x200000, 0x200000, 0xfbd3cd90 )
ROM_END

ROM_START( nwarru )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vphu.03f", 0x000000, 0x80000, 0x85d6a359 )
	ROM_LOAD16_WORD_SWAP( "vphu.04c", 0x080000, 0x80000, 0xcb7fce77 )
	ROM_LOAD16_WORD_SWAP( "vphu.05e", 0x100000, 0x80000, 0xe08f2bba )
	ROM_LOAD16_WORD_SWAP( "vphu.06c", 0x180000, 0x80000, 0x08c04cdb )
	ROM_LOAD16_WORD_SWAP( "vphu.07b", 0x200000, 0x80000, 0xb5a5ab19 )
	ROM_LOAD16_WORD_SWAP( "vphu.08b", 0x280000, 0x80000, 0x51bb20fb )
	ROM_LOAD16_WORD_SWAP( "vphu.09b", 0x300000, 0x80000, 0x41a64205 )
	ROM_LOAD16_WORD_SWAP( "vphu.10b", 0x380000, 0x80000, 0x2b1d43ae )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vph.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.14m",   0x1000000, 0x400000, 0x7a0e1add , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.16m",   0x1000002, 0x400000, 0x2f41ca75 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.18m",   0x1000004, 0x400000, 0x64498eed , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.20m",   0x1000006, 0x400000, 0x17f2433f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vph.01",   0x00000, 0x08000, 0x5045dcac )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vph.02",   0x28000, 0x20000, 0x86b60e59 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vph.11m",   0x000000, 0x200000, 0xe1837d33 )
	ROM_LOAD16_WORD_SWAP( "vph.12m",   0x200000, 0x200000, 0xfbd3cd90 )
ROM_END

ROM_START( nwarrh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vphh.03d", 0x000000, 0x80000, 0x6029c7be )
	ROM_LOAD16_WORD_SWAP( "vphh.04a", 0x080000, 0x80000, 0xd26625ee )
	ROM_LOAD16_WORD_SWAP( "vphh.05c", 0x100000, 0x80000, 0x73ee0b8a )
	ROM_LOAD16_WORD_SWAP( "vphh.06a", 0x180000, 0x80000, 0xa5b3a50a )
	ROM_LOAD16_WORD_SWAP( "vphh.07",  0x200000, 0x80000, 0x5fc2bdc1 )
	ROM_LOAD16_WORD_SWAP( "vphh.08",  0x280000, 0x80000, 0xe65588d9 )
	ROM_LOAD16_WORD_SWAP( "vphh.09",  0x300000, 0x80000, 0xa2ce6d63 )
	ROM_LOAD16_WORD_SWAP( "vphh.10",  0x380000, 0x80000, 0xe2f4f4b9 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vph.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.14m",   0x1000000, 0x400000, 0x7a0e1add , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.16m",   0x1000002, 0x400000, 0x2f41ca75 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.18m",   0x1000004, 0x400000, 0x64498eed , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.20m",   0x1000006, 0x400000, 0x17f2433f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vph.01",   0x00000, 0x08000, 0x5045dcac )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vph.02",   0x28000, 0x20000, 0x86b60e59 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vph.11m",   0x000000, 0x200000, 0xe1837d33 )
	ROM_LOAD16_WORD_SWAP( "vph.12m",   0x200000, 0x200000, 0xfbd3cd90 )
ROM_END

ROM_START( nwarrb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vphb.03d", 0x000000, 0x80000, 0x3a426d3f )
	ROM_LOAD16_WORD_SWAP( "vphb.04a", 0x080000, 0x80000, 0x51c4bb2f )
	ROM_LOAD16_WORD_SWAP( "vphb.05c", 0x100000, 0x80000, 0xac44d997 )
	ROM_LOAD16_WORD_SWAP( "vphb.06a", 0x180000, 0x80000, 0x5072a5fe )
	ROM_LOAD16_WORD_SWAP( "vphb.07",  0x200000, 0x80000, 0x9b355192 )
	ROM_LOAD16_WORD_SWAP( "vphb.08",  0x280000, 0x80000, 0x42220f84 )
	ROM_LOAD16_WORD_SWAP( "vphb.09",  0x300000, 0x80000, 0x029e015d )
	ROM_LOAD16_WORD_SWAP( "vphb.10",  0x380000, 0x80000, 0x37b3ce37 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vph.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.14m",   0x1000000, 0x400000, 0x7a0e1add , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.16m",   0x1000002, 0x400000, 0x2f41ca75 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.18m",   0x1000004, 0x400000, 0x64498eed , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.20m",   0x1000006, 0x400000, 0x17f2433f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vph.01",   0x00000, 0x08000, 0x5045dcac )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vph.02",   0x28000, 0x20000, 0x86b60e59 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vph.11m",   0x000000, 0x200000, 0xe1837d33 )
	ROM_LOAD16_WORD_SWAP( "vph.12m",   0x200000, 0x200000, 0xfbd3cd90 )
ROM_END

ROM_START( vhuntj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vphj.03f", 0x000000, 0x80000, 0x3de2e333 )
	ROM_LOAD16_WORD_SWAP( "vphj.04c", 0x080000, 0x80000, 0xc95cf304 )
	ROM_LOAD16_WORD_SWAP( "vphj.05d", 0x100000, 0x80000, 0x50de5ddd )
	ROM_LOAD16_WORD_SWAP( "vphj.06c", 0x180000, 0x80000, 0xac3bd3d5 )
	ROM_LOAD16_WORD_SWAP( "vphj.07b", 0x200000, 0x80000, 0x0761309f )
	ROM_LOAD16_WORD_SWAP( "vphj.08b", 0x280000, 0x80000, 0x5a5c2bf5 )
	ROM_LOAD16_WORD_SWAP( "vphj.09b", 0x300000, 0x80000, 0x823d6d99 )
	ROM_LOAD16_WORD_SWAP( "vphj.10b", 0x380000, 0x80000, 0x32c7d8f0 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vph.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.14m",   0x1000000, 0x400000, 0x7a0e1add , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.16m",   0x1000002, 0x400000, 0x2f41ca75 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.18m",   0x1000004, 0x400000, 0x64498eed , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.20m",   0x1000006, 0x400000, 0x17f2433f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vph.01",   0x00000, 0x08000, 0x5045dcac )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vph.02",   0x28000, 0x20000, 0x86b60e59 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vph.11m",   0x000000, 0x200000, 0xe1837d33 )
	ROM_LOAD16_WORD_SWAP( "vph.12m",   0x200000, 0x200000, 0xfbd3cd90 )
ROM_END

ROM_START( vhuntjr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vphj.03c", 0x000000, 0x80000, 0x606b682a )
	ROM_LOAD16_WORD_SWAP( "vphj.04b", 0x080000, 0x80000, 0xa3b40393 )
	ROM_LOAD16_WORD_SWAP( "vphj.05b", 0x100000, 0x80000, 0xfccd5558 )
	ROM_LOAD16_WORD_SWAP( "vphj.06b", 0x180000, 0x80000, 0x07e10a73 )
	ROM_LOAD16_WORD_SWAP( "vphj.07b", 0x200000, 0x80000, 0x0761309f )
	ROM_LOAD16_WORD_SWAP( "vphj.08b", 0x280000, 0x80000, 0x5a5c2bf5 )
	ROM_LOAD16_WORD_SWAP( "vphj.09b", 0x300000, 0x80000, 0x823d6d99 )
	ROM_LOAD16_WORD_SWAP( "vphj.10b", 0x380000, 0x80000, 0x32c7d8f0 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vph.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.14m",   0x1000000, 0x400000, 0x7a0e1add , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.16m",   0x1000002, 0x400000, 0x2f41ca75 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.18m",   0x1000004, 0x400000, 0x64498eed , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.20m",   0x1000006, 0x400000, 0x17f2433f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vph.01",   0x00000, 0x08000, 0x5045dcac )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vph.02",   0x28000, 0x20000, 0x86b60e59 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vph.11m",   0x000000, 0x200000, 0xe1837d33 )
	ROM_LOAD16_WORD_SWAP( "vph.12m",   0x200000, 0x200000, 0xfbd3cd90 )
ROM_END

ROM_START( vhuntjr2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vphj.03b", 0x000000, 0x80000, 0x679c3fa9 )
	ROM_LOAD16_WORD_SWAP( "vphj.04a", 0x080000, 0x80000, 0xeb6e71e4 )
	ROM_LOAD16_WORD_SWAP( "vphj.05a", 0x100000, 0x80000, 0xeaf634ea )
	ROM_LOAD16_WORD_SWAP( "vphj.06a", 0x180000, 0x80000, 0xb70cc6be )
	ROM_LOAD16_WORD_SWAP( "vphj.07a", 0x200000, 0x80000, 0x46ab907d )
	ROM_LOAD16_WORD_SWAP( "vphj.08a", 0x280000, 0x80000, 0x1c00355e )
	ROM_LOAD16_WORD_SWAP( "vphj.09a", 0x300000, 0x80000, 0x026e6f82 )
	ROM_LOAD16_WORD_SWAP( "vphj.10a", 0x380000, 0x80000, 0xaadfb3ea )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vph.13m",   0x0000000, 0x400000, 0xc51baf99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.15m",   0x0000002, 0x400000, 0x3ce83c77 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.17m",   0x0000004, 0x400000, 0x4f2408e0 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.19m",   0x0000006, 0x400000, 0x9ff60250 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.14m",   0x1000000, 0x400000, 0x7a0e1add , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.16m",   0x1000002, 0x400000, 0x2f41ca75 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.18m",   0x1000004, 0x400000, 0x64498eed , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vph.20m",   0x1000006, 0x400000, 0x17f2433f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vph.01",   0x00000, 0x08000, 0x5045dcac )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vph.02",   0x28000, 0x20000, 0x86b60e59 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vph.11m",   0x000000, 0x200000, 0xe1837d33 )
	ROM_LOAD16_WORD_SWAP( "vph.12m",   0x200000, 0x200000, 0xfbd3cd90 )
ROM_END

ROM_START( progear )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pgau.03", 0x000000, 0x80000, 0x343a783e )
	ROM_LOAD16_WORD_SWAP( "pgau.04", 0x080000, 0x80000, 0x16208d79 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "pga-simm.01c",   0x0000000, 0x200000,  0x452f98b0 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.01d",   0x0000001, 0x200000,  0x9e672092 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.01a",   0x0000002, 0x200000,  0xae9ddafe , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.01b",   0x0000003, 0x200000,  0x94d72d94 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03c",   0x0000004, 0x200000,  0x48a1886d , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03d",   0x0000005, 0x200000,  0x172d7e37 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03a",   0x0000006, 0x200000,  0x9ee33d98 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03b",   0x0000007, 0x200000,  0x848dee32 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pga.01",   0x00000, 0x08000, 0xbdbfa992 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pga-simm.05a",   0x000000, 0x200000, 0xc0aac80c ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pga-simm.05b",   0x200000, 0x200000, 0x37a65d86 ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pga-simm.06a",   0x400000, 0x200000, 0xd3f1e934 ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pga-simm.06b",   0x600000, 0x200000, 0x8b39489a ) // ROM on a simm
ROM_END

ROM_START( progearj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pgaj.03", 0x000000, 0x80000, 0x06dbba54 )
	ROM_LOAD16_WORD_SWAP( "pgaj.04", 0x080000, 0x80000, 0xa1f1f1bc )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "pga-simm.01c",   0x0000000, 0x200000,  0x452f98b0 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.01d",   0x0000001, 0x200000,  0x9e672092 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.01a",   0x0000002, 0x200000,  0xae9ddafe , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.01b",   0x0000003, 0x200000,  0x94d72d94 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03c",   0x0000004, 0x200000,  0x48a1886d , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03d",   0x0000005, 0x200000,  0x172d7e37 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03a",   0x0000006, 0x200000,  0x9ee33d98 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03b",   0x0000007, 0x200000,  0x848dee32 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pga.01",   0x00000, 0x08000, 0xbdbfa992 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pga-simm.05a",   0x000000, 0x200000, 0xc0aac80c ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pga-simm.05b",   0x200000, 0x200000, 0x37a65d86 ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pga-simm.06a",   0x400000, 0x200000, 0xd3f1e934 ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pga-simm.06b",   0x600000, 0x200000, 0x8b39489a ) // ROM on a simm
ROM_END

ROM_START( progeara )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pgaa.03", 0x000000, 0x80000, 0x25e6e2ce )
	ROM_LOAD16_WORD_SWAP( "pgaa.04", 0x080000, 0x80000, 0x8104307e )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "pga-simm.01c",   0x0000000, 0x200000,  0x452f98b0 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.01d",   0x0000001, 0x200000,  0x9e672092 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.01a",   0x0000002, 0x200000,  0xae9ddafe , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.01b",   0x0000003, 0x200000,  0x94d72d94 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03c",   0x0000004, 0x200000,  0x48a1886d , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03d",   0x0000005, 0x200000,  0x172d7e37 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03a",   0x0000006, 0x200000,  0x9ee33d98 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pga-simm.03b",   0x0000007, 0x200000,  0x848dee32 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pga.01",   0x00000, 0x08000, 0xbdbfa992 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pga-simm.05a",   0x000000, 0x200000, 0xc0aac80c ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pga-simm.05b",   0x200000, 0x200000, 0x37a65d86 ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pga-simm.06a",   0x400000, 0x200000, 0xd3f1e934 ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pga-simm.06b",   0x600000, 0x200000, 0x8b39489a ) // ROM on a simm
ROM_END

ROM_START( pzloop2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pl2e.03", 0x000000, 0x80000, 0x3b1285b2 )
	ROM_LOAD16_WORD_SWAP( "pl2e.04", 0x080000, 0x80000, 0x40a2d647 )
	ROM_LOAD16_WORD_SWAP( "pl2e.05", 0x100000, 0x80000, 0x0f11d818 )
	ROM_LOAD16_WORD_SWAP( "pl2e.06", 0x180000, 0x80000, 0x86fbbdf4 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "pl2-simm.01c",   0x0000000, 0x200000, 0x137b13a7 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.01d",   0x0000001, 0x200000, 0xa2db1507 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.01a",   0x0000002, 0x200000, 0x7e80ff8e , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.01b",   0x0000003, 0x200000, 0xcd93e6ed , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.03c",   0x0000004, 0x200000, 0x0f52bbca , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.03d",   0x0000005, 0x200000, 0xa62712c3 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.03a",   0x0000006, 0x200000, 0xb60c9f8e , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.03b",   0x0000007, 0x200000, 0x83fef284 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pl2.01",   0x00000, 0x08000, 0x35697569 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pl2-simm.05a",   0x000000, 0x200000, 0x85d8fbe8 ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pl2-simm.05b",   0x200000, 0x200000, 0x1ed62584 ) // ROM on a simm
ROM_END

ROM_START( pzloop2j )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pl2j.03a", 0x000000, 0x80000, 0x0a751bd0 )
	ROM_LOAD16_WORD_SWAP( "pl2j.04a", 0x080000, 0x80000, 0xc3f72afe )
	ROM_LOAD16_WORD_SWAP( "pl2j.05a", 0x100000, 0x80000, 0x6ea9dbfc )
	ROM_LOAD16_WORD_SWAP( "pl2j.06a", 0x180000, 0x80000, 0x0f14848d )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "pl2-simm.01c",   0x0000000, 0x200000, 0x137b13a7 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.01d",   0x0000001, 0x200000, 0xa2db1507 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.01a",   0x0000002, 0x200000, 0x7e80ff8e , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.01b",   0x0000003, 0x200000, 0xcd93e6ed , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.03c",   0x0000004, 0x200000, 0x0f52bbca , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.03d",   0x0000005, 0x200000, 0xa62712c3 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.03a",   0x0000006, 0x200000, 0xb60c9f8e , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm
	ROMX_LOAD( "pl2-simm.03b",   0x0000007, 0x200000, 0x83fef284 , ROM_GROUPBYTE | ROM_SKIP(7) ) // ROM on a simm

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pl2.01",   0x00000, 0x08000, 0x35697569 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pl2-simm.05a",   0x000000, 0x200000, 0x85d8fbe8 ) // ROM on a simm
	ROM_LOAD16_WORD_SWAP( "pl2-simm.05b",   0x200000, 0x200000, 0x1ed62584 ) // ROM on a simm
ROM_END

ROM_START( qndream )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "tqzj.03a", 0x000000, 0x80000, 0x7acf3e30 )
	ROM_LOAD16_WORD_SWAP( "tqzj.04",  0x080000, 0x80000, 0xf1044a87 )
	ROM_LOAD16_WORD_SWAP( "tqzj.05",  0x100000, 0x80000, 0x4105ba0e )
	ROM_LOAD16_WORD_SWAP( "tqzj.06",  0x180000, 0x80000, 0xc371e8a5 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "tqz.14m",   0x800000, 0x200000, 0x98af88a2 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "tqz.16m",   0x800002, 0x200000, 0xdf82d491 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "tqz.18m",   0x800004, 0x200000, 0x42f132ff , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "tqz.20m",   0x800006, 0x200000, 0xb2e128a3 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "tqz.01",   0x00000, 0x08000, 0xe9ce9d0a )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "tqz.11m",   0x000000, 0x200000, 0x78e7884f )
	ROM_LOAD16_WORD_SWAP( "tqz.12m",   0x200000, 0x200000, 0x2e049b13 )
ROM_END

ROM_START( ringdest )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "smbe.03b", 0x000000, 0x80000, 0xb8016278 )
	ROM_LOAD16_WORD_SWAP( "smbe.04b", 0x080000, 0x80000, 0x18c4c447 )
	ROM_LOAD16_WORD_SWAP( "smbe.05b", 0x100000, 0x80000, 0x18ebda7f )
	ROM_LOAD16_WORD_SWAP( "smbe.06b", 0x180000, 0x80000, 0x89c80007 )
	ROM_LOAD16_WORD_SWAP( "smb.07",   0x200000, 0x80000, 0xb9a11577 )
	ROM_LOAD16_WORD_SWAP( "smb.08",   0x280000, 0x80000, 0xf931b76b )

	ROM_REGION( 0x1200000, REGION_GFX1, 0 )
	ROMX_LOAD( "smb.13m",   0x0000000, 0x200000, 0xd9b2d1de , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.15m",   0x0000002, 0x200000, 0x9a766d92 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.17m",   0x0000004, 0x200000, 0x51800f0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.19m",   0x0000006, 0x200000, 0x35757e96 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.14m",   0x0800000, 0x200000, 0xe5bfd0e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.16m",   0x0800002, 0x200000, 0xc56c0866 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.18m",   0x0800004, 0x200000, 0x4ded3910 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.20m",   0x0800006, 0x200000, 0x26ea1ec5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.21m",   0x1000000, 0x080000, 0x0a08c5fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.23m",   0x1000002, 0x080000, 0x0911b6c4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.25m",   0x1000004, 0x080000, 0x82d6c4ec , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.27m",   0x1000006, 0x080000, 0x9b48678b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "smb.01",   0x00000, 0x08000, 0x0abc229a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "smb.02",   0x28000, 0x20000, 0xd051679a )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "smb.11m",   0x000000, 0x200000, 0xc56935f9 )
	ROM_LOAD16_WORD_SWAP( "smb.12m",   0x200000, 0x200000, 0x955b0782 )
ROM_END

ROM_START( smbomb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "smbj.03a", 0x000000, 0x80000, 0x1c5613de )
	ROM_LOAD16_WORD_SWAP( "smbj.04a", 0x080000, 0x80000, 0x29071ed7 )
	ROM_LOAD16_WORD_SWAP( "smbj.05a", 0x100000, 0x80000, 0xeb20bce4 )
	ROM_LOAD16_WORD_SWAP( "smbj.06a", 0x180000, 0x80000, 0x94b420cd )
	ROM_LOAD16_WORD_SWAP( "smb.07",  0x200000, 0x80000, 0xb9a11577 )
	ROM_LOAD16_WORD_SWAP( "smb.08",  0x280000, 0x80000, 0xf931b76b )

	ROM_REGION( 0x1200000, REGION_GFX1, 0 )
	ROMX_LOAD( "smb.13m",   0x0000000, 0x200000, 0xd9b2d1de , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.15m",   0x0000002, 0x200000, 0x9a766d92 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.17m",   0x0000004, 0x200000, 0x51800f0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.19m",   0x0000006, 0x200000, 0x35757e96 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.14m",   0x0800000, 0x200000, 0xe5bfd0e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.16m",   0x0800002, 0x200000, 0xc56c0866 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.18m",   0x0800004, 0x200000, 0x4ded3910 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.20m",   0x0800006, 0x200000, 0x26ea1ec5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.21m",   0x1000000, 0x080000, 0x0a08c5fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.23m",   0x1000002, 0x080000, 0x0911b6c4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.25m",   0x1000004, 0x080000, 0x82d6c4ec , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.27m",   0x1000006, 0x080000, 0x9b48678b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "smb.01",   0x00000, 0x08000, 0x0abc229a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "smb.02",   0x28000, 0x20000, 0xd051679a )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "smb.11m",   0x000000, 0x200000, 0xc56935f9 )
	ROM_LOAD16_WORD_SWAP( "smb.12m",   0x200000, 0x200000, 0x955b0782 )
ROM_END

ROM_START( smbombr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "smbj.03", 0x000000, 0x80000, 0x52eafb10 )
	ROM_LOAD16_WORD_SWAP( "smbj.04", 0x080000, 0x80000, 0xaa6e8078 )
	ROM_LOAD16_WORD_SWAP( "smbj.05", 0x100000, 0x80000, 0xb69e7d5f )
	ROM_LOAD16_WORD_SWAP( "smbj.06", 0x180000, 0x80000, 0x8d857b56 )
	ROM_LOAD16_WORD_SWAP( "smb.07",  0x200000, 0x80000, 0xb9a11577 )
	ROM_LOAD16_WORD_SWAP( "smb.08",  0x280000, 0x80000, 0xf931b76b )

	ROM_REGION( 0x1200000, REGION_GFX1, 0 )
	ROMX_LOAD( "smb.13m",   0x0000000, 0x200000, 0xd9b2d1de , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.15m",   0x0000002, 0x200000, 0x9a766d92 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.17m",   0x0000004, 0x200000, 0x51800f0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.19m",   0x0000006, 0x200000, 0x35757e96 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.14m",   0x0800000, 0x200000, 0xe5bfd0e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.16m",   0x0800002, 0x200000, 0xc56c0866 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.18m",   0x0800004, 0x200000, 0x4ded3910 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.20m",   0x0800006, 0x200000, 0x26ea1ec5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.21m",   0x1000000, 0x080000, 0x0a08c5fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.23m",   0x1000002, 0x080000, 0x0911b6c4 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.25m",   0x1000004, 0x080000, 0x82d6c4ec , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "smb.27m",   0x1000006, 0x080000, 0x9b48678b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "smb.01",   0x00000, 0x08000, 0x0abc229a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "smb.02",   0x28000, 0x20000, 0xd051679a )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "smb.11m",   0x000000, 0x200000, 0xc56935f9 )
	ROM_LOAD16_WORD_SWAP( "smb.12m",   0x200000, 0x200000, 0x955b0782 )
ROM_END

ROM_START( mmancp2u )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "rcmu.03b", 0x000000, 0x80000, 0xc39f037f )
	ROM_LOAD16_WORD_SWAP( "rcmu.04b", 0x080000, 0x80000, 0xcd6f5e99 )
	ROM_LOAD16_WORD_SWAP( "rcm.05b",  0x100000, 0x80000, 0x4376ea95 ) // == rcma_21a.rom from CPS1 version

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "rcm.73",   0x800000, 0x80000, 0x774c6e04 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.63",   0x800002, 0x80000, 0xacad7c62 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.83",   0x800004, 0x80000, 0x6af30499 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.93",   0x800006, 0x80000, 0x7a5a5166 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.74",   0xa00000, 0x80000, 0x004ec725 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.64",   0xa00002, 0x80000, 0x65c0464e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.84",   0xa00004, 0x80000, 0xfb3097cc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.94",   0xa00006, 0x80000, 0x2e16557a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.75",   0xc00000, 0x80000, 0x70a73f99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.65",   0xc00002, 0x80000, 0xecedad3d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.85",   0xc00004, 0x80000, 0x3d6186d8 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.95",   0xc00006, 0x80000, 0x8c7700f1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.76",   0xe00000, 0x80000, 0x89a889ad , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.66",   0xe00002, 0x80000, 0x1300eb7b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.86",   0xe00004, 0x80000, 0x6d974ebd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.96",   0xe00006, 0x80000, 0x7da4cd24 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "rcm.01",   0x00000, 0x08000, 0xd60cf8a3 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "rcm.51",   0x000000, 0x80000, 0xb6d07080 )
	ROM_LOAD16_WORD_SWAP( "rcm.52",   0x080000, 0x80000, 0xdfddc493 )
	ROM_LOAD16_WORD_SWAP( "rcm.53",   0x100000, 0x80000, 0x6062ae3a )
	ROM_LOAD16_WORD_SWAP( "rcm.54",   0x180000, 0x80000, 0x08c6f3bf )
	ROM_LOAD16_WORD_SWAP( "rcm.55",   0x200000, 0x80000, 0xf97dfccc )
	ROM_LOAD16_WORD_SWAP( "rcm.56",   0x280000, 0x80000, 0xade475bc )
	ROM_LOAD16_WORD_SWAP( "rcm.57",   0x300000, 0x80000, 0x075effb3 )
	ROM_LOAD16_WORD_SWAP( "rcm.58",   0x380000, 0x80000, 0xf6c1f87b )
ROM_END

ROM_START( rmancp2j )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "rcmj.03a", 0x000000, 0x80000, 0x30559f60 )
	ROM_LOAD16_WORD_SWAP( "rcmj.04a", 0x080000, 0x80000, 0x5efc9366 )
	ROM_LOAD16_WORD_SWAP( "rcm.05a", 0x100000, 0x80000, 0x517ccde2 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "rcm.73",   0x800000, 0x80000, 0x774c6e04 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.63",   0x800002, 0x80000, 0xacad7c62 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.83",   0x800004, 0x80000, 0x6af30499 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.93",   0x800006, 0x80000, 0x7a5a5166 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.74",   0xa00000, 0x80000, 0x004ec725 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.64",   0xa00002, 0x80000, 0x65c0464e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.84",   0xa00004, 0x80000, 0xfb3097cc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.94",   0xa00006, 0x80000, 0x2e16557a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.75",   0xc00000, 0x80000, 0x70a73f99 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.65",   0xc00002, 0x80000, 0xecedad3d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.85",   0xc00004, 0x80000, 0x3d6186d8 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.95",   0xc00006, 0x80000, 0x8c7700f1 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.76",   0xe00000, 0x80000, 0x89a889ad , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.66",   0xe00002, 0x80000, 0x1300eb7b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.86",   0xe00004, 0x80000, 0x6d974ebd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "rcm.96",   0xe00006, 0x80000, 0x7da4cd24 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "rcm.01",   0x00000, 0x08000, 0xd60cf8a3 )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "rcm.51",   0x000000, 0x80000, 0xb6d07080 )
	ROM_LOAD16_WORD_SWAP( "rcm.52",   0x080000, 0x80000, 0xdfddc493 )
	ROM_LOAD16_WORD_SWAP( "rcm.53",   0x100000, 0x80000, 0x6062ae3a )
	ROM_LOAD16_WORD_SWAP( "rcm.54",   0x180000, 0x80000, 0x08c6f3bf )
	ROM_LOAD16_WORD_SWAP( "rcm.55",   0x200000, 0x80000, 0xf97dfccc )
	ROM_LOAD16_WORD_SWAP( "rcm.56",   0x280000, 0x80000, 0xade475bc )
	ROM_LOAD16_WORD_SWAP( "rcm.57",   0x300000, 0x80000, 0x075effb3 )
	ROM_LOAD16_WORD_SWAP( "rcm.58",   0x380000, 0x80000, 0xf6c1f87b )
ROM_END

ROM_START( sfa )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfze.03d", 0x000000, 0x80000, 0xebf2054d )
	ROM_LOAD16_WORD_SWAP( "sfz.04b",  0x080000, 0x80000, 0x8b73b0e5 )
	ROM_LOAD16_WORD_SWAP( "sfz.05a",  0x100000, 0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfar1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfze.03c", 0x000000, 0x80000, 0xa1b69dd7 )  // Rom name dosnt appear to follow normal capcom naming system and was written on rom by hand
	ROM_LOAD16_WORD_SWAP( "sfze.04b", 0x080000, 0x80000, 0xbb90acd5 )  // Rom name dosnt appear to follow normal capcom naming system and was written on rom by hand
	ROM_LOAD16_WORD_SWAP( "sfz.05a",  0x100000, 0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfar2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfze.03b", 0x000000, 0x80000, 0x2bf5708e )  // Rom name dosnt appear to follow normal capcom naming system and was written on rom by hand
	ROM_LOAD16_WORD_SWAP( "sfz.04a",  0x080000, 0x80000, 0x5f99e9a5 )
	ROM_LOAD16_WORD_SWAP( "sfz.05a",  0x100000, 0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfar3 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfze.03a", 0x000000, 0x80000, 0xfdbcd434 )
	ROM_LOAD16_WORD_SWAP( "sfz.04",   0x080000, 0x80000, 0x0c436d30 )
	ROM_LOAD16_WORD_SWAP( "sfz.05",   0x100000, 0x80000, 0x1f363612 )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfau )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfzu.03a", 0x000000, 0x80000, 0x49fc7db9 )
	ROM_LOAD16_WORD_SWAP( "sfz.04a",  0x080000, 0x80000, 0x5f99e9a5 )
	ROM_LOAD16_WORD_SWAP( "sfz.05a",  0x100000, 0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfza )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfza.03a", 0x000000, 0x80000, 0xca91bed9 )
	ROM_LOAD16_WORD_SWAP( "sfz.04a",  0x080000, 0x80000, 0x5f99e9a5 )
	ROM_LOAD16_WORD_SWAP( "sfz.05a",  0x100000, 0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfzj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfzj.03c", 0x000000, 0x80000, 0xf5444120 )
	ROM_LOAD16_WORD_SWAP( "sfz.04b",  0x080000, 0x80000, 0x8b73b0e5 )
	ROM_LOAD16_WORD_SWAP( "sfz.05a",  0x100000, 0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfzjr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfzj.03b", 0x000000, 0x80000, 0x844220c2 )
	ROM_LOAD16_WORD_SWAP( "sfz.04a",  0x080000, 0x80000, 0x5f99e9a5 )
	ROM_LOAD16_WORD_SWAP( "sfz.05a",  0x100000, 0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfzjr2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfzj.03a", 0x000000, 0x80000, 0x3cfce93c )
	ROM_LOAD16_WORD_SWAP( "sfz.04",   0x080000, 0x80000, 0x0c436d30 )
	ROM_LOAD16_WORD_SWAP( "sfz.05",   0x100000, 0x80000, 0x1f363612 )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfzh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfzh.03c", 0x000000, 0x80000, 0xbce635aa )
	ROM_LOAD16_WORD_SWAP( "sfz.04a",  0x080000, 0x80000, 0x5f99e9a5 )
	ROM_LOAD16_WORD_SWAP( "sfz.05a",  0x100000, 0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",   0x180000, 0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfzb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfzb.03g", 0x000000, 0x80000, 0x348862d4 )
	ROM_LOAD16_WORD_SWAP( "sfzb.04e", 0x080000, 0x80000, 0x8d9b2480 )
	ROM_LOAD16_WORD_SWAP( "sfz.05a", 0x100000,  0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",  0x180000,  0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfzbr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfzb.03e", 0x000000, 0x80000, 0xecba89a3 )
	ROM_LOAD16_WORD_SWAP( "sfz.04b",  0x080000, 0x80000, 0x8b73b0e5 )
	ROM_LOAD16_WORD_SWAP( "sfz.05a", 0x100000,  0x80000, 0x0810544d )
	ROM_LOAD16_WORD_SWAP( "sfz.06",  0x180000,  0x80000, 0x806e8f38 )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROM_FILL(              0x000000, 0x800000, 0 )
	ROMX_LOAD( "sfz.14m",   0x800000, 0x200000, 0x90fefdb3 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.16m",   0x800002, 0x200000, 0x5354c948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.18m",   0x800004, 0x200000, 0x41a1e790 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfz.20m",   0x800006, 0x200000, 0xa549df98 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfz.01",   0x00000, 0x08000, 0xffffec7d )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfz.02",   0x28000, 0x20000, 0x45f46a08 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfz.11m",   0x000000, 0x200000, 0xc4b093cd )
	ROM_LOAD16_WORD_SWAP( "sfz.12m",   0x200000, 0x200000, 0x8bdbc4b4 )
ROM_END

ROM_START( sfa2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz2u.03", 0x000000, 0x80000, 0x84a09006 )
	ROM_LOAD16_WORD_SWAP( "sz2u.04", 0x080000, 0x80000, 0xac46e5ed )
	ROM_LOAD16_WORD_SWAP( "sz2u.05", 0x100000, 0x80000, 0x6c0c79d3 )
	ROM_LOAD16_WORD_SWAP( "sz2u.06", 0x180000, 0x80000, 0xc5c8eb63 )
	ROM_LOAD16_WORD_SWAP( "sz2u.07", 0x200000, 0x80000, 0x5de01cc5 )
	ROM_LOAD16_WORD_SWAP( "sz2u.08", 0x280000, 0x80000, 0xbea11d56 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2j )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz2j.03a", 0x000000, 0x80000, 0x97461e28 )
	ROM_LOAD16_WORD_SWAP( "sz2j.04a", 0x080000, 0x80000, 0xae4851a9 )
	ROM_LOAD16_WORD_SWAP( "sz2.05",   0x100000, 0x80000, 0x98e8e992 )
	ROM_LOAD16_WORD_SWAP( "sz2.06",   0x180000, 0x80000, 0x5b1d49c0 )
	ROM_LOAD16_WORD_SWAP( "sz2j.07a", 0x200000, 0x80000, 0xd910b2a2 )
	ROM_LOAD16_WORD_SWAP( "sz2.08",   0x280000, 0x80000, 0x0fe8585d )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2a )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz2a.03a", 0x000000, 0x80000, 0x30d2099f )
	ROM_LOAD16_WORD_SWAP( "sz2a.04a", 0x080000, 0x80000, 0x1cc94db1 )
	ROM_LOAD16_WORD_SWAP( "sz2.05",   0x100000, 0x80000, 0x98e8e992 )
	ROM_LOAD16_WORD_SWAP( "sz2.06",   0x180000, 0x80000, 0x5b1d49c0 )
	ROM_LOAD16_WORD_SWAP( "sz2a.07a", 0x200000, 0x80000, 0x0aed2494 )
	ROM_LOAD16_WORD_SWAP( "sz2.08",   0x280000, 0x80000, 0x0fe8585d )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2b )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz2b.03b", 0x000000, 0x80000, 0x1ac12812 )
	ROM_LOAD16_WORD_SWAP( "sz2b.04b", 0x080000, 0x80000, 0xe4ffaf68 )
	ROM_LOAD16_WORD_SWAP( "sz2b.05a", 0x100000, 0x80000, 0xdd224156 )
	ROM_LOAD16_WORD_SWAP( "sz2b.06a", 0x180000, 0x80000, 0xa45a75a6 )
	ROM_LOAD16_WORD_SWAP( "sz2b.07a", 0x200000, 0x80000, 0x7d19d5ec )
	ROM_LOAD16_WORD_SWAP( "sz2b.08",  0x280000, 0x80000, 0x92b66e01 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2br1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz2b.03", 0x000000, 0x80000, 0xe6ce530b )
	ROM_LOAD16_WORD_SWAP( "sz2b.04", 0x080000, 0x80000, 0x1605a0cb )
	ROM_LOAD16_WORD_SWAP( "sz2b.05", 0x100000, 0x80000, 0x4b442a7c )
	ROM_LOAD16_WORD_SWAP( "sz2.06",  0x180000, 0x80000, 0x5b1d49c0 )
	ROM_LOAD16_WORD_SWAP( "sz2b.07", 0x200000, 0x80000, 0x947e8ac6 )
	ROM_LOAD16_WORD_SWAP( "sz2b.08", 0x280000, 0x80000, 0x92b66e01 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2h )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz2h.03", 0x000000, 0x80000, 0xbfeddf5b )
	ROM_LOAD16_WORD_SWAP( "sz2h.04", 0x080000, 0x80000, 0xea5009fb )
	ROM_LOAD16_WORD_SWAP( "sz2b.05", 0x100000, 0x80000, 0x4b442a7c )
	ROM_LOAD16_WORD_SWAP( "sz2.06",  0x180000, 0x80000, 0x5b1d49c0 )
	ROM_LOAD16_WORD_SWAP( "sz2b.07", 0x200000, 0x80000, 0x947e8ac6 )
	ROM_LOAD16_WORD_SWAP( "sz2b.08", 0x280000, 0x80000, 0x92b66e01 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2n )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz2n.03", 0x000000, 0x80000, 0x58924741 )
	ROM_LOAD16_WORD_SWAP( "sz2n.04", 0x080000, 0x80000, 0x592a17c5 )
	ROM_LOAD16_WORD_SWAP( "sz2b.05", 0x100000, 0x80000, 0x4b442a7c )
	ROM_LOAD16_WORD_SWAP( "sz2.06",  0x180000, 0x80000, 0x5b1d49c0 )
	ROM_LOAD16_WORD_SWAP( "sz2n.07", 0x200000, 0x80000, 0x8e184246 )
	ROM_LOAD16_WORD_SWAP( "sz2.08",  0x280000, 0x80000, 0x0fe8585d )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2aj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "szaj.03a", 0x000000, 0x80000, 0xa3802fe3 )
	ROM_LOAD16_WORD_SWAP( "szaj.04a", 0x080000, 0x80000, 0xe7ca87c7 )
	ROM_LOAD16_WORD_SWAP( "szaj.05a", 0x100000, 0x80000, 0xc88ebf88 )
	ROM_LOAD16_WORD_SWAP( "szaj.06a", 0x180000, 0x80000, 0x35ed5b7a )
	ROM_LOAD16_WORD_SWAP( "szaj.07a", 0x200000, 0x80000, 0x975dcb3e )
	ROM_LOAD16_WORD_SWAP( "szaj.08a", 0x280000, 0x80000, 0xdc73f2d7 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2aa )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "szaa.03", 0x000000, 0x80000, 0x88e7023e )
	ROM_LOAD16_WORD_SWAP( "szaa.04", 0x080000, 0x80000, 0xae8ec36e )
	ROM_LOAD16_WORD_SWAP( "szaa.05", 0x100000, 0x80000, 0xf053a55e )
	ROM_LOAD16_WORD_SWAP( "szaa.06", 0x180000, 0x80000, 0xcfc0e7a8 )
	ROM_LOAD16_WORD_SWAP( "szaa.07", 0x200000, 0x80000, 0x5feb8b20 )
	ROM_LOAD16_WORD_SWAP( "szaa.08", 0x280000, 0x80000, 0x6eb6d412 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2ah )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "szah.03", 0x000000, 0x80000, 0x06f93d1d )
	ROM_LOAD16_WORD_SWAP( "szah.04", 0x080000, 0x80000, 0xe62ee914 )
	ROM_LOAD16_WORD_SWAP( "szah.05", 0x100000, 0x80000, 0x2b7f4b20 )
	ROM_LOAD16_WORD_SWAP( "sza.06",  0x180000, 0x80000, 0x0abda2fc )
	ROM_LOAD16_WORD_SWAP( "sza.07",  0x200000, 0x80000, 0xe9430762 )
	ROM_LOAD16_WORD_SWAP( "sza.08",  0x280000, 0x80000, 0xb65711a9 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfz2ab )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "szab.03", 0x000000, 0x80000, 0xcb436eca )
	ROM_LOAD16_WORD_SWAP( "szab.04", 0x080000, 0x80000, 0x14534bea )
	ROM_LOAD16_WORD_SWAP( "szab.05", 0x100000, 0x80000, 0x7fb10658 )
	ROM_LOAD16_WORD_SWAP( "sza.06",  0x180000, 0x80000, 0x0abda2fc )
	ROM_LOAD16_WORD_SWAP( "sza.07",  0x200000, 0x80000, 0xe9430762 )
	ROM_LOAD16_WORD_SWAP( "sza.08",  0x280000, 0x80000, 0xb65711a9 )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz2.13m",   0x0000000, 0x400000, 0x4d1f1f22 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.15m",   0x0000002, 0x400000, 0x19cea680 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.17m",   0x0000004, 0x400000, 0xe01b4588 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.19m",   0x0000006, 0x400000, 0x0feeda64 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.14m",   0x1000000, 0x100000, 0x0560c6aa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.16m",   0x1000002, 0x100000, 0xae940f87 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.18m",   0x1000004, 0x100000, 0x4bc3c8bc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz2.20m",   0x1000006, 0x100000, 0x39e674c0 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz2.01a",   0x00000, 0x08000, 0x1bc323cf )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz2.02a",   0x28000, 0x20000, 0xba6a5013 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz2.11m",   0x000000, 0x200000, 0xaa47a601 )
	ROM_LOAD16_WORD_SWAP( "sz2.12m",   0x200000, 0x200000, 0x2237bc53 )
ROM_END

ROM_START( sfa3 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz3e_03c.03", 0x000000, 0x80000, 0x9762b206 )
	ROM_LOAD16_WORD_SWAP( "sz3e_04c.04", 0x080000, 0x80000, 0x5ad3f721 )
	ROM_LOAD16_WORD_SWAP( "sz3.05c",  0x100000, 0x80000, 0x57fd0a40 )
	ROM_LOAD16_WORD_SWAP( "sz3.06c",  0x180000, 0x80000, 0xf6305f8b )
	ROM_LOAD16_WORD_SWAP( "sz3.07c",  0x200000, 0x80000, 0x6eab0f6f )
	ROM_LOAD16_WORD_SWAP( "sz3.08c",  0x280000, 0x80000, 0x910c4a3b )
	ROM_LOAD16_WORD_SWAP( "sz3.09c",  0x300000, 0x80000, 0xb29e5199 )
	ROM_LOAD16_WORD_SWAP( "sz3.10b",  0x380000, 0x80000, 0xdeb2ff52 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz3.13m",   0x0000000, 0x400000, 0x0f7a60d9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.15m",   0x0000002, 0x400000, 0x8e933741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.17m",   0x0000004, 0x400000, 0xd6e98147 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.19m",   0x0000006, 0x400000, 0xf31a728a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.14m",   0x1000000, 0x400000, 0x5ff98297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.16m",   0x1000002, 0x400000, 0x52b5bdee , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.18m",   0x1000004, 0x400000, 0x40631ed5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.20m",   0x1000006, 0x400000, 0x763409b4 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz3.01",   0x00000, 0x08000, 0xde810084 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz3.02",   0x28000, 0x20000, 0x72445dc4 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz3.11m",   0x000000, 0x400000, 0x1c89eed1 )
	ROM_LOAD16_WORD_SWAP( "sz3.12m",   0x400000, 0x400000, 0xf392b13a )
ROM_END

ROM_START( sfa3u )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz3u.03c", 0x000000, 0x80000, 0xe007da2e )
	ROM_LOAD16_WORD_SWAP( "sz3u.04c", 0x080000, 0x80000, 0x5f78f0e7 )
	ROM_LOAD16_WORD_SWAP( "sz3.05c",  0x100000, 0x80000, 0x57fd0a40 )
	ROM_LOAD16_WORD_SWAP( "sz3.06c",  0x180000, 0x80000, 0xf6305f8b )
	ROM_LOAD16_WORD_SWAP( "sz3.07c",  0x200000, 0x80000, 0x6eab0f6f )
	ROM_LOAD16_WORD_SWAP( "sz3.08c",  0x280000, 0x80000, 0x910c4a3b )
	ROM_LOAD16_WORD_SWAP( "sz3.09c",  0x300000, 0x80000, 0xb29e5199 )
	ROM_LOAD16_WORD_SWAP( "sz3.10b",  0x380000, 0x80000, 0xdeb2ff52 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz3.13m",   0x0000000, 0x400000, 0x0f7a60d9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.15m",   0x0000002, 0x400000, 0x8e933741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.17m",   0x0000004, 0x400000, 0xd6e98147 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.19m",   0x0000006, 0x400000, 0xf31a728a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.14m",   0x1000000, 0x400000, 0x5ff98297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.16m",   0x1000002, 0x400000, 0x52b5bdee , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.18m",   0x1000004, 0x400000, 0x40631ed5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.20m",   0x1000006, 0x400000, 0x763409b4 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz3.01",   0x00000, 0x08000, 0xde810084 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz3.02",   0x28000, 0x20000, 0x72445dc4 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz3.11m",   0x000000, 0x400000, 0x1c89eed1 )
	ROM_LOAD16_WORD_SWAP( "sz3.12m",   0x400000, 0x400000, 0xf392b13a )
ROM_END

ROM_START( sfa3ur1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz3u.03", 0x000000, 0x80000, 0xb5984a19 )
	ROM_LOAD16_WORD_SWAP( "sz3u.04", 0x080000, 0x80000, 0x7e8158ba )
	ROM_LOAD16_WORD_SWAP( "sz3.05",  0x100000, 0x80000, 0x9b21518a )
	ROM_LOAD16_WORD_SWAP( "sz3.06",  0x180000, 0x80000, 0xe7a6c3a7 )
	ROM_LOAD16_WORD_SWAP( "sz3.07",  0x200000, 0x80000, 0xec4c0cfd )
	ROM_LOAD16_WORD_SWAP( "sz3.08",  0x280000, 0x80000, 0x5c7e7240 )
	ROM_LOAD16_WORD_SWAP( "sz3.09",  0x300000, 0x80000, 0xc5589553 )
	ROM_LOAD16_WORD_SWAP( "sz3.10",  0x380000, 0x80000, 0xa9717252 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz3.13m",   0x0000000, 0x400000, 0x0f7a60d9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.15m",   0x0000002, 0x400000, 0x8e933741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.17m",   0x0000004, 0x400000, 0xd6e98147 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.19m",   0x0000006, 0x400000, 0xf31a728a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.14m",   0x1000000, 0x400000, 0x5ff98297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.16m",   0x1000002, 0x400000, 0x52b5bdee , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.18m",   0x1000004, 0x400000, 0x40631ed5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.20m",   0x1000006, 0x400000, 0x763409b4 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz3.01",   0x00000, 0x08000, 0xde810084 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz3.02",   0x28000, 0x20000, 0x72445dc4 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz3.11m",   0x000000, 0x400000, 0x1c89eed1 )
	ROM_LOAD16_WORD_SWAP( "sz3.12m",   0x400000, 0x400000, 0xf392b13a )
ROM_END

ROM_START( sfz3j )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz3j.03c", 0x000000, 0x80000, 0xcadf4a51 )
	ROM_LOAD16_WORD_SWAP( "sz3j.04c", 0x080000, 0x80000, 0xfcb31228 )
	ROM_LOAD16_WORD_SWAP( "sz3.05c",  0x100000, 0x80000, 0x57fd0a40 )
	ROM_LOAD16_WORD_SWAP( "sz3.06c",  0x180000, 0x80000, 0xf6305f8b )
	ROM_LOAD16_WORD_SWAP( "sz3.07c",  0x200000, 0x80000, 0x6eab0f6f )
	ROM_LOAD16_WORD_SWAP( "sz3.08c",  0x280000, 0x80000, 0x910c4a3b )
	ROM_LOAD16_WORD_SWAP( "sz3.09c",  0x300000, 0x80000, 0xb29e5199 )
	ROM_LOAD16_WORD_SWAP( "sz3.10b",  0x380000, 0x80000, 0xdeb2ff52 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz3.13m",   0x0000000, 0x400000, 0x0f7a60d9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.15m",   0x0000002, 0x400000, 0x8e933741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.17m",   0x0000004, 0x400000, 0xd6e98147 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.19m",   0x0000006, 0x400000, 0xf31a728a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.14m",   0x1000000, 0x400000, 0x5ff98297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.16m",   0x1000002, 0x400000, 0x52b5bdee , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.18m",   0x1000004, 0x400000, 0x40631ed5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.20m",   0x1000006, 0x400000, 0x763409b4 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz3.01",   0x00000, 0x08000, 0xde810084 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz3.02",   0x28000, 0x20000, 0x72445dc4 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz3.11m",   0x000000, 0x400000, 0x1c89eed1 )
	ROM_LOAD16_WORD_SWAP( "sz3.12m",   0x400000, 0x400000, 0xf392b13a )
ROM_END

ROM_START( sfz3jr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz3j.03a", 0x000000, 0x80000, 0x6ee0beae )
	ROM_LOAD16_WORD_SWAP( "sz3j.04a", 0x080000, 0x80000, 0xa6e2978d )
	ROM_LOAD16_WORD_SWAP( "sz3.05a",  0x100000, 0x80000, 0x05964b7d )
	ROM_LOAD16_WORD_SWAP( "sz3.06a",  0x180000, 0x80000, 0x78ce2179 )
	ROM_LOAD16_WORD_SWAP( "sz3.07a",  0x200000, 0x80000, 0x398bf52f )
	ROM_LOAD16_WORD_SWAP( "sz3.08a",  0x280000, 0x80000, 0x866d0588 )
	ROM_LOAD16_WORD_SWAP( "sz3.09a",  0x300000, 0x80000, 0x2180892c )
	ROM_LOAD16_WORD_SWAP( "sz3.10",   0x380000, 0x80000, 0xa9717252 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz3.13m",   0x0000000, 0x400000, 0x0f7a60d9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.15m",   0x0000002, 0x400000, 0x8e933741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.17m",   0x0000004, 0x400000, 0xd6e98147 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.19m",   0x0000006, 0x400000, 0xf31a728a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.14m",   0x1000000, 0x400000, 0x5ff98297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.16m",   0x1000002, 0x400000, 0x52b5bdee , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.18m",   0x1000004, 0x400000, 0x40631ed5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.20m",   0x1000006, 0x400000, 0x763409b4 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz3.01",   0x00000, 0x08000, 0xde810084 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz3.02",   0x28000, 0x20000, 0x72445dc4 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz3.11m",   0x000000, 0x400000, 0x1c89eed1 )
	ROM_LOAD16_WORD_SWAP( "sz3.12m",   0x400000, 0x400000, 0xf392b13a )
ROM_END

ROM_START( sfz3jr2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz3j.03", 0x000000, 0x80000, 0xf7cb4b13 )
	ROM_LOAD16_WORD_SWAP( "sz3j.04", 0x080000, 0x80000, 0x0846c29d )
	ROM_LOAD16_WORD_SWAP( "sz3.05",  0x100000, 0x80000, 0x9b21518a )
	ROM_LOAD16_WORD_SWAP( "sz3.06",  0x180000, 0x80000, 0xe7a6c3a7 )
	ROM_LOAD16_WORD_SWAP( "sz3.07",  0x200000, 0x80000, 0xec4c0cfd )
	ROM_LOAD16_WORD_SWAP( "sz3.08",  0x280000, 0x80000, 0x5c7e7240 )
	ROM_LOAD16_WORD_SWAP( "sz3.09",  0x300000, 0x80000, 0xc5589553 )
	ROM_LOAD16_WORD_SWAP( "sz3.10",  0x380000, 0x80000, 0xa9717252 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz3.13m",   0x0000000, 0x400000, 0x0f7a60d9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.15m",   0x0000002, 0x400000, 0x8e933741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.17m",   0x0000004, 0x400000, 0xd6e98147 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.19m",   0x0000006, 0x400000, 0xf31a728a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.14m",   0x1000000, 0x400000, 0x5ff98297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.16m",   0x1000002, 0x400000, 0x52b5bdee , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.18m",   0x1000004, 0x400000, 0x40631ed5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.20m",   0x1000006, 0x400000, 0x763409b4 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz3.01",   0x00000, 0x08000, 0xde810084 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz3.02",   0x28000, 0x20000, 0x72445dc4 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz3.11m",   0x000000, 0x400000, 0x1c89eed1 )
	ROM_LOAD16_WORD_SWAP( "sz3.12m",   0x400000, 0x400000, 0xf392b13a )
ROM_END

ROM_START( sfz3a )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz3a.03d", 0x000000, 0x80000, 0xd7e140d6 )
	ROM_LOAD16_WORD_SWAP( "sz3a.04d", 0x080000, 0x80000, 0xe06869a2 )
	ROM_LOAD16_WORD_SWAP( "sz3.05c",  0x100000, 0x80000, 0x57fd0a40 )
	ROM_LOAD16_WORD_SWAP( "sz3.06c",  0x180000, 0x80000, 0xf6305f8b )
	ROM_LOAD16_WORD_SWAP( "sz3.07c",  0x200000, 0x80000, 0x6eab0f6f )
	ROM_LOAD16_WORD_SWAP( "sz3.08c",  0x280000, 0x80000, 0x910c4a3b )
	ROM_LOAD16_WORD_SWAP( "sz3.09c",  0x300000, 0x80000, 0xb29e5199 )
	ROM_LOAD16_WORD_SWAP( "sz3.10b",  0x380000, 0x80000, 0xdeb2ff52 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz3.13m",   0x0000000, 0x400000, 0x0f7a60d9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.15m",   0x0000002, 0x400000, 0x8e933741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.17m",   0x0000004, 0x400000, 0xd6e98147 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.19m",   0x0000006, 0x400000, 0xf31a728a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.14m",   0x1000000, 0x400000, 0x5ff98297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.16m",   0x1000002, 0x400000, 0x52b5bdee , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.18m",   0x1000004, 0x400000, 0x40631ed5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.20m",   0x1000006, 0x400000, 0x763409b4 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz3.01",   0x00000, 0x08000, 0xde810084 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz3.02",   0x28000, 0x20000, 0x72445dc4 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz3.11m",   0x000000, 0x400000, 0x1c89eed1 )
	ROM_LOAD16_WORD_SWAP( "sz3.12m",   0x400000, 0x400000, 0xf392b13a )
ROM_END

ROM_START( sfz3ar1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz3a.03a", 0x000000, 0x80000, 0x29c681fd )
	ROM_LOAD16_WORD_SWAP( "sz3a.04",  0x080000, 0x80000, 0x9ddd1484 )
	ROM_LOAD16_WORD_SWAP( "sz3.05",   0x100000, 0x80000, 0x9b21518a )
	ROM_LOAD16_WORD_SWAP( "sz3.06",   0x180000, 0x80000, 0xe7a6c3a7 )
	ROM_LOAD16_WORD_SWAP( "sz3.07",   0x200000, 0x80000, 0xec4c0cfd )
	ROM_LOAD16_WORD_SWAP( "sz3.08",   0x280000, 0x80000, 0x5c7e7240 )
	ROM_LOAD16_WORD_SWAP( "sz3.09",   0x300000, 0x80000, 0xc5589553 )
	ROM_LOAD16_WORD_SWAP( "sz3.10",   0x380000, 0x80000, 0xa9717252 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz3.13m",   0x0000000, 0x400000, 0x0f7a60d9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.15m",   0x0000002, 0x400000, 0x8e933741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.17m",   0x0000004, 0x400000, 0xd6e98147 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.19m",   0x0000006, 0x400000, 0xf31a728a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.14m",   0x1000000, 0x400000, 0x5ff98297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.16m",   0x1000002, 0x400000, 0x52b5bdee , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.18m",   0x1000004, 0x400000, 0x40631ed5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.20m",   0x1000006, 0x400000, 0x763409b4 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz3.01",   0x00000, 0x08000, 0xde810084 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz3.02",   0x28000, 0x20000, 0x72445dc4 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz3.11m",   0x000000, 0x400000, 0x1c89eed1 )
	ROM_LOAD16_WORD_SWAP( "sz3.12m",   0x400000, 0x400000, 0xf392b13a )
ROM_END

ROM_START( sfa3b )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sz3b.03", 0x000000, 0x80000, 0x046c9b4d )
	ROM_LOAD16_WORD_SWAP( "sz3b.04", 0x080000, 0x80000, 0xda211919 )
	ROM_LOAD16_WORD_SWAP( "sz3.05",  0x100000, 0x80000, 0x9b21518a )
	ROM_LOAD16_WORD_SWAP( "sz3.06",  0x180000, 0x80000, 0xe7a6c3a7 )
	ROM_LOAD16_WORD_SWAP( "sz3.07",  0x200000, 0x80000, 0xec4c0cfd )
	ROM_LOAD16_WORD_SWAP( "sz3.08",  0x280000, 0x80000, 0x5c7e7240 )
	ROM_LOAD16_WORD_SWAP( "sz3.09",  0x300000, 0x80000, 0xc5589553 )
	ROM_LOAD16_WORD_SWAP( "sz3.10",  0x380000, 0x80000, 0xa9717252 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "sz3.13m",   0x0000000, 0x400000, 0x0f7a60d9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.15m",   0x0000002, 0x400000, 0x8e933741 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.17m",   0x0000004, 0x400000, 0xd6e98147 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.19m",   0x0000006, 0x400000, 0xf31a728a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.14m",   0x1000000, 0x400000, 0x5ff98297 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.16m",   0x1000002, 0x400000, 0x52b5bdee , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.18m",   0x1000004, 0x400000, 0x40631ed5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sz3.20m",   0x1000006, 0x400000, 0x763409b4 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sz3.01",   0x00000, 0x08000, 0xde810084 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sz3.02",   0x28000, 0x20000, 0x72445dc4 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sz3.11m",   0x000000, 0x400000, 0x1c89eed1 )
	ROM_LOAD16_WORD_SWAP( "sz3.12m",   0x400000, 0x400000, 0xf392b13a )
ROM_END

ROM_START( sgemf )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pcfu.03", 0x000000, 0x80000, 0xac2e8566 )
	ROM_LOAD16_WORD_SWAP( "pcf.04",  0x080000, 0x80000, 0xf4314c96 )
	ROM_LOAD16_WORD_SWAP( "pcf.05",  0x100000, 0x80000, 0x215655f6 )
	ROM_LOAD16_WORD_SWAP( "pcf.06",  0x180000, 0x80000, 0xea6f13ea )
	ROM_LOAD16_WORD_SWAP( "pcf.07",  0x200000, 0x80000, 0x5ac6d5ea )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pcf.13m",   0x0000000, 0x400000, 0x22d72ab9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.15m",   0x0000002, 0x400000, 0x16a4813c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.17m",   0x0000004, 0x400000, 0x1097e035 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.19m",   0x0000006, 0x400000, 0xd362d874 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.14m",   0x1000000, 0x100000, 0x0383897c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.16m",   0x1000002, 0x100000, 0x76f91084 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.18m",   0x1000004, 0x100000, 0x756c3754 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.20m",   0x1000006, 0x100000, 0x9ec9277d , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pcf.01",   0x00000, 0x08000, 0x254e5f33 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pcf.02",   0x28000, 0x20000, 0x6902f4f9 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pcf.11m",   0x000000, 0x400000, 0xa5dea005 )
	ROM_LOAD16_WORD_SWAP( "pcf.12m",   0x400000, 0x400000, 0x4ce235fe )
ROM_END

ROM_START( pfghtj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pcfj.03", 0x000000, 0x80000, 0x681da43e )
	ROM_LOAD16_WORD_SWAP( "pcf.04",  0x080000, 0x80000, 0xf4314c96 )
	ROM_LOAD16_WORD_SWAP( "pcf.05",  0x100000, 0x80000, 0x215655f6 )
	ROM_LOAD16_WORD_SWAP( "pcf.06",  0x180000, 0x80000, 0xea6f13ea )
	ROM_LOAD16_WORD_SWAP( "pcf.07",  0x200000, 0x80000, 0x5ac6d5ea )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pcf.13m",   0x0000000, 0x400000, 0x22d72ab9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.15m",   0x0000002, 0x400000, 0x16a4813c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.17m",   0x0000004, 0x400000, 0x1097e035 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.19m",   0x0000006, 0x400000, 0xd362d874 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.14m",   0x1000000, 0x100000, 0x0383897c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.16m",   0x1000002, 0x100000, 0x76f91084 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.18m",   0x1000004, 0x100000, 0x756c3754 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.20m",   0x1000006, 0x100000, 0x9ec9277d , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pcf.01",   0x00000, 0x08000, 0x254e5f33 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pcf.02",   0x28000, 0x20000, 0x6902f4f9 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pcf.11m",   0x000000, 0x400000, 0xa5dea005 )
	ROM_LOAD16_WORD_SWAP( "pcf.12m",   0x400000, 0x400000, 0x4ce235fe )
ROM_END

ROM_START( sgemfa )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pcfa.03", 0x000000, 0x80000, 0xe17c089a )
	ROM_LOAD16_WORD_SWAP( "pcf.04",  0x080000, 0x80000, 0xf4314c96 )
	ROM_LOAD16_WORD_SWAP( "pcf.05",  0x100000, 0x80000, 0x215655f6 )
	ROM_LOAD16_WORD_SWAP( "pcf.06",  0x180000, 0x80000, 0xea6f13ea )
	ROM_LOAD16_WORD_SWAP( "pcf.07",  0x200000, 0x80000, 0x5ac6d5ea )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pcf.13m",   0x0000000, 0x400000, 0x22d72ab9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.15m",   0x0000002, 0x400000, 0x16a4813c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.17m",   0x0000004, 0x400000, 0x1097e035 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.19m",   0x0000006, 0x400000, 0xd362d874 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.14m",   0x1000000, 0x100000, 0x0383897c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.16m",   0x1000002, 0x100000, 0x76f91084 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.18m",   0x1000004, 0x100000, 0x756c3754 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.20m",   0x1000006, 0x100000, 0x9ec9277d , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pcf.01",   0x00000, 0x08000, 0x254e5f33 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pcf.02",   0x28000, 0x20000, 0x6902f4f9 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pcf.11m",   0x000000, 0x400000, 0xa5dea005 )
	ROM_LOAD16_WORD_SWAP( "pcf.12m",   0x400000, 0x400000, 0x4ce235fe )
ROM_END

ROM_START( sgemfh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pcfh.03", 0x000000, 0x80000, 0xe9103347 )
	ROM_LOAD16_WORD_SWAP( "pcf.04",  0x080000, 0x80000, 0xf4314c96 )
	ROM_LOAD16_WORD_SWAP( "pcf.05",  0x100000, 0x80000, 0x215655f6 )
	ROM_LOAD16_WORD_SWAP( "pcf.06",  0x180000, 0x80000, 0xea6f13ea )
	ROM_LOAD16_WORD_SWAP( "pcf.07",  0x200000, 0x80000, 0x5ac6d5ea )

	ROM_REGION( 0x1400000, REGION_GFX1, 0 )
	ROMX_LOAD( "pcf.13m",   0x0000000, 0x400000, 0x22d72ab9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.15m",   0x0000002, 0x400000, 0x16a4813c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.17m",   0x0000004, 0x400000, 0x1097e035 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.19m",   0x0000006, 0x400000, 0xd362d874 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.14m",   0x1000000, 0x100000, 0x0383897c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.16m",   0x1000002, 0x100000, 0x76f91084 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.18m",   0x1000004, 0x100000, 0x756c3754 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pcf.20m",   0x1000006, 0x100000, 0x9ec9277d , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pcf.01",   0x00000, 0x08000, 0x254e5f33 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pcf.02",   0x28000, 0x20000, 0x6902f4f9 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pcf.11m",   0x000000, 0x400000, 0xa5dea005 )
	ROM_LOAD16_WORD_SWAP( "pcf.12m",   0x400000, 0x400000, 0x4ce235fe )
ROM_END

ROM_START( spf2t )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pzfu.03a", 0x000000, 0x80000, 0x346e62ef )
	ROM_LOAD16_WORD_SWAP( "pzf.04",   0x080000, 0x80000, 0xb80649e2 )

	ROM_REGION( 0xC00000, REGION_GFX1, 0 )
	ROM_FILL(             0x000000, 0x800000, 0 )
	ROMX_LOAD( "pzf.14m",  0x800000, 0x100000, 0x2d4881cb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pzf.16m",  0x800002, 0x100000, 0x4b0fd1be , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pzf.18m",  0x800004, 0x100000, 0xe43aac33 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pzf.20m",  0x800006, 0x100000, 0x7f536ff1 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pzf.01",   0x00000, 0x08000, 0x600fb2a3 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pzf.02",   0x28000, 0x20000, 0x496076e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pzf.11m",   0x000000, 0x200000, 0x78442743 )
	ROM_LOAD16_WORD_SWAP( "pzf.12m",   0x200000, 0x200000, 0x399d2c7b )
ROM_END

ROM_START( spf2xj )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pzfj.03a", 0x000000, 0x80000, 0x2070554a )
	ROM_LOAD16_WORD_SWAP( "pzf.04",   0x080000, 0x80000, 0xb80649e2 )

	ROM_REGION( 0xC00000, REGION_GFX1, 0 )
	ROM_FILL(             0x000000, 0x800000, 0 )
	ROMX_LOAD( "pzf.14m",  0x800000, 0x100000, 0x2d4881cb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pzf.16m",  0x800002, 0x100000, 0x4b0fd1be , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pzf.18m",  0x800004, 0x100000, 0xe43aac33 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pzf.20m",  0x800006, 0x100000, 0x7f536ff1 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pzf.01",   0x00000, 0x08000, 0x600fb2a3 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pzf.02",   0x28000, 0x20000, 0x496076e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pzf.11m",   0x000000, 0x200000, 0x78442743 )
	ROM_LOAD16_WORD_SWAP( "pzf.12m",   0x200000, 0x200000, 0x399d2c7b )
ROM_END

ROM_START( spf2ta )
	ROM_REGION(CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "pzfa.03",  0x000000, 0x80000, 0x3cecfa78 )
	ROM_LOAD16_WORD_SWAP( "pzf.04",   0x080000, 0x80000, 0xb80649e2 )

	ROM_REGION( 0xC00000, REGION_GFX1, 0 )
	ROM_FILL(             0x000000, 0x800000, 0 )
	ROMX_LOAD( "pzf.14m",  0x800000, 0x100000, 0x2d4881cb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pzf.16m",  0x800002, 0x100000, 0x4b0fd1be , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pzf.18m",  0x800004, 0x100000, 0xe43aac33 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "pzf.20m",  0x800006, 0x100000, 0x7f536ff1 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION(QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "pzf.01",   0x00000, 0x08000, 0x600fb2a3 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "pzf.02",   0x28000, 0x20000, 0x496076e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "pzf.11m",   0x000000, 0x200000, 0x78442743 )
	ROM_LOAD16_WORD_SWAP( "pzf.12m",   0x200000, 0x200000, 0x399d2c7b )
ROM_END

ROM_START( ssf2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfe.03",  0x000000, 0x80000, 0xa597745d )
	ROM_LOAD16_WORD_SWAP( "ssfe.04",  0x080000, 0x80000, 0xb082aa67 )
	ROM_LOAD16_WORD_SWAP( "ssfe.05",  0x100000, 0x80000, 0x02b9c137 )
	ROM_LOAD16_WORD_SWAP( "ssfe.06",  0x180000, 0x80000, 0x70d470c5 )
	ROM_LOAD16_WORD_SWAP( "ssfe.07",  0x200000, 0x80000, 0x2409001d )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2u )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfu.03a", 0x000000, 0x80000, 0x72f29c33 )
	ROM_LOAD16_WORD_SWAP( "ssfu.04a", 0x080000, 0x80000, 0x935cea44 )
	ROM_LOAD16_WORD_SWAP( "ssfu.05",  0x100000, 0x80000, 0xa0acb28a )
	ROM_LOAD16_WORD_SWAP( "ssfu.06",  0x180000, 0x80000, 0x47413dcf )
	ROM_LOAD16_WORD_SWAP( "ssfu.07",  0x200000, 0x80000, 0xe6066077 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2a )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfa.03b", 0x000000, 0x80000, 0x83a059bf )
	ROM_LOAD16_WORD_SWAP( "ssfa.04a", 0x080000, 0x80000, 0x5d873642 )
	ROM_LOAD16_WORD_SWAP( "ssfa.05",  0x100000, 0x80000, 0xf8fb4de2 )
	ROM_LOAD16_WORD_SWAP( "ssfa.06b", 0x180000, 0x80000, 0x3185d19d )
	ROM_LOAD16_WORD_SWAP( "ssfa.07",  0x200000, 0x80000, 0x36e29217 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2ar1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfa.03a", 0x000000, 0x80000, 0xd2a3c520 )
	ROM_LOAD16_WORD_SWAP( "ssfa.04a", 0x080000, 0x80000, 0x5d873642 )
	ROM_LOAD16_WORD_SWAP( "ssfa.05",  0x100000, 0x80000, 0xf8fb4de2 )
	ROM_LOAD16_WORD_SWAP( "ssfa.06", 0x180000, 0x80000, 0xaa8acee7 )
	ROM_LOAD16_WORD_SWAP( "ssfa.07",  0x200000, 0x80000, 0x36e29217 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2j )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfj.03b", 0x000000, 0x80000, 0x5c2e356d )
	ROM_LOAD16_WORD_SWAP( "ssfj.04a", 0x080000, 0x80000, 0x013bd55c )
	ROM_LOAD16_WORD_SWAP( "ssfj.05",  0x100000, 0x80000, 0x0918d19a )
	ROM_LOAD16_WORD_SWAP( "ssfj.06b", 0x180000, 0x80000, 0x014e0c6d )
	ROM_LOAD16_WORD_SWAP( "ssfj.07",  0x200000, 0x80000, 0xeb6a9b1b )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2jr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfj.03a", 0x000000, 0x80000, 0x0bbf1304 )
	ROM_LOAD16_WORD_SWAP( "ssfj.04a", 0x080000, 0x80000, 0x013bd55c )
	ROM_LOAD16_WORD_SWAP( "ssfj.05",  0x100000, 0x80000, 0x0918d19a )
	ROM_LOAD16_WORD_SWAP( "ssfj.06",  0x180000, 0x80000, 0xd808a6cd )
	ROM_LOAD16_WORD_SWAP( "ssfj.07",  0x200000, 0x80000, 0xeb6a9b1b )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2jr2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfj.03", 0x000000, 0x80000, 0x7eb0efed )
	ROM_LOAD16_WORD_SWAP( "ssfj.04", 0x080000, 0x80000, 0xd7322164 )
	ROM_LOAD16_WORD_SWAP( "ssfj.05", 0x100000, 0x80000, 0x0918d19a )
	ROM_LOAD16_WORD_SWAP( "ssfj.06", 0x180000, 0x80000, 0xd808a6cd )
	ROM_LOAD16_WORD_SWAP( "ssfj.07", 0x200000, 0x80000, 0xeb6a9b1b )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2tb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfe.3tc", 0x000000, 0x80000, 0x496a8409 )
	ROM_LOAD16_WORD_SWAP( "ssfe.4tc", 0x080000, 0x80000, 0x4b45c18b )
	ROM_LOAD16_WORD_SWAP( "ssfe.5t",  0x100000, 0x80000, 0x6a9c6444 )
	ROM_LOAD16_WORD_SWAP( "ssfe.6tb", 0x180000, 0x80000, 0xe4944fc3 )
	ROM_LOAD16_WORD_SWAP( "ssfe.7t",  0x200000, 0x80000, 0x2c9f4782 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2tbr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfe.3t", 0x000000, 0x80000, 0x1e018e34 )
	ROM_LOAD16_WORD_SWAP( "ssfe.4t", 0x080000, 0x80000, 0xac92efaf )
	ROM_LOAD16_WORD_SWAP( "ssfe.5t", 0x100000, 0x80000, 0x6a9c6444 )
	ROM_LOAD16_WORD_SWAP( "ssfe.6t", 0x180000, 0x80000, 0xf442562b )
	ROM_LOAD16_WORD_SWAP( "ssfe.7t", 0x200000, 0x80000, 0x2c9f4782 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2tbj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "ssfj.3t", 0x000000, 0x80000, 0x980d4450 )
	ROM_LOAD16_WORD_SWAP( "ssfj.4t", 0x080000, 0x80000, 0xb4dc1906 )
	ROM_LOAD16_WORD_SWAP( "ssfj.5t", 0x100000, 0x80000, 0xa7e35fbc )
	ROM_LOAD16_WORD_SWAP( "ssfj.6t", 0x180000, 0x80000, 0xcb592b30 )
	ROM_LOAD16_WORD_SWAP( "ssfj.7t", 0x200000, 0x80000, 0x1f239515 )

	ROM_REGION( 0xc00000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ssf.01",   0x00000, 0x08000, 0xeb247e8c )
	ROM_CONTINUE(         0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD( "ssf.q01",  0x000000, 0x080000, 0xa6f9da5c )
	ROM_LOAD( "ssf.q02",  0x080000, 0x080000, 0x8c66ae26 )
	ROM_LOAD( "ssf.q03",  0x100000, 0x080000, 0x695cc2ca )
	ROM_LOAD( "ssf.q04",  0x180000, 0x080000, 0x9d9ebe32 )
	ROM_LOAD( "ssf.q05",  0x200000, 0x080000, 0x4770e7b7 )
	ROM_LOAD( "ssf.q06",  0x280000, 0x080000, 0x4e79c951 )
	ROM_LOAD( "ssf.q07",  0x300000, 0x080000, 0xcdd14313 )
	ROM_LOAD( "ssf.q08",  0x380000, 0x080000, 0x6f5a088c )
ROM_END

ROM_START( ssf2t )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfxe.03c", 0x000000, 0x80000, 0x2fa1f396 )
	ROM_LOAD16_WORD_SWAP( "sfxe.04a", 0x080000, 0x80000, 0xd0bc29c6 )
	ROM_LOAD16_WORD_SWAP( "sfxe.05",  0x100000, 0x80000, 0x65222964 )
	ROM_LOAD16_WORD_SWAP( "sfxe.06a", 0x180000, 0x80000, 0x8fe9f531 )
	ROM_LOAD16_WORD_SWAP( "sfxe.07",  0x200000, 0x80000, 0x8a7d0cb6 )
	ROM_LOAD16_WORD_SWAP( "sfxe.08",  0x280000, 0x80000, 0x74c24062 )
	ROM_LOAD16_WORD_SWAP( "sfx.09",   0x300000, 0x80000, 0x642fae3f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.21m",   0xc00000, 0x100000, 0xe32854af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.23m",   0xc00002, 0x100000, 0x760f2927 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.25m",   0xc00004, 0x100000, 0x1ee90208 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.27m",   0xc00006, 0x100000, 0xf814400f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfx.01",   0x00000, 0x08000, 0xb47b8835 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfx.02",   0x28000, 0x20000, 0x0022633f )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfx.11m",   0x000000, 0x200000, 0x9bdbd476 )
	ROM_LOAD16_WORD_SWAP( "sfx.12m",   0x200000, 0x200000, 0xa05e3aab )
ROM_END

ROM_START( ssf2tu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfxu.03e", 0x000000, 0x80000, 0xd6ff689e )
	ROM_LOAD16_WORD_SWAP( "sfxu.04a", 0x080000, 0x80000, 0x532b5ffd )
	ROM_LOAD16_WORD_SWAP( "sfxu.05",  0x100000, 0x80000, 0xffa3c6de )
	ROM_LOAD16_WORD_SWAP( "sfxu.06b", 0x180000, 0x80000, 0x83f9382b )
	ROM_LOAD16_WORD_SWAP( "sfxu.07a", 0x200000, 0x80000, 0x6ab673e8 )
	ROM_LOAD16_WORD_SWAP( "sfxu.08",  0x280000, 0x80000, 0xb3c71810 )
	ROM_LOAD16_WORD_SWAP( "sfx.09",   0x300000, 0x80000, 0x642fae3f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.21m",   0xc00000, 0x100000, 0xe32854af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.23m",   0xc00002, 0x100000, 0x760f2927 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.25m",   0xc00004, 0x100000, 0x1ee90208 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.27m",   0xc00006, 0x100000, 0xf814400f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfx.01",   0x00000, 0x08000, 0xb47b8835 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfx.02",   0x28000, 0x20000, 0x0022633f )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfx.11m",   0x000000, 0x200000, 0x9bdbd476 )
	ROM_LOAD16_WORD_SWAP( "sfx.12m",   0x200000, 0x200000, 0xa05e3aab )
ROM_END

ROM_START( ssf2tur1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfxu.03c", 0x000000, 0x80000, 0x86e4a335 )
	ROM_LOAD16_WORD_SWAP( "sfxu.04a", 0x080000, 0x80000, 0x532b5ffd )
	ROM_LOAD16_WORD_SWAP( "sfxu.05",  0x100000, 0x80000, 0xffa3c6de )
	ROM_LOAD16_WORD_SWAP( "sfxu.06a", 0x180000, 0x80000, 0xe4c04c99 )
	ROM_LOAD16_WORD_SWAP( "sfxu.07",  0x200000, 0x80000, 0xd8199e41 )
	ROM_LOAD16_WORD_SWAP( "sfxu.08",  0x280000, 0x80000, 0xb3c71810 )
	ROM_LOAD16_WORD_SWAP( "sfx.09",   0x300000, 0x80000, 0x642fae3f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.21m",   0xc00000, 0x100000, 0xe32854af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.23m",   0xc00002, 0x100000, 0x760f2927 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.25m",   0xc00004, 0x100000, 0x1ee90208 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.27m",   0xc00006, 0x100000, 0xf814400f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfx.01",   0x00000, 0x08000, 0xb47b8835 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfx.02",   0x28000, 0x20000, 0x0022633f )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfx.11m",   0x000000, 0x200000, 0x9bdbd476 )
	ROM_LOAD16_WORD_SWAP( "sfx.12m",   0x200000, 0x200000, 0xa05e3aab )
ROM_END

ROM_START( ssf2ta )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfxa.03c", 0x000000, 0x80000, 0x04b9ff34 )
	ROM_LOAD16_WORD_SWAP( "sfxa.04a", 0x080000, 0x80000, 0x16ea5f7a )
	ROM_LOAD16_WORD_SWAP( "sfxa.05",  0x100000, 0x80000, 0x53d61f0c )
	ROM_LOAD16_WORD_SWAP( "sfxa.06a", 0x180000, 0x80000, 0x066d09b5 )
	ROM_LOAD16_WORD_SWAP( "sfxa.07",  0x200000, 0x80000, 0xa428257b )
	ROM_LOAD16_WORD_SWAP( "sfxa.08",  0x280000, 0x80000, 0x39be596c )
	ROM_LOAD16_WORD_SWAP( "sfx.09",   0x300000, 0x80000, 0x642fae3f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.21m",   0xc00000, 0x100000, 0xe32854af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.23m",   0xc00002, 0x100000, 0x760f2927 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.25m",   0xc00004, 0x100000, 0x1ee90208 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.27m",   0xc00006, 0x100000, 0xf814400f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfx.01",   0x00000, 0x08000, 0xb47b8835 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfx.02",   0x28000, 0x20000, 0x0022633f )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfx.11m",   0x000000, 0x200000, 0x9bdbd476 )
	ROM_LOAD16_WORD_SWAP( "sfx.12m",   0x200000, 0x200000, 0xa05e3aab )
ROM_END

ROM_START( ssf2xj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "sfxj.03c", 0x000000, 0x80000, 0xa7417b79 )
	ROM_LOAD16_WORD_SWAP( "sfxj.04a", 0x080000, 0x80000, 0xaf7767b4 )
	ROM_LOAD16_WORD_SWAP( "sfxj.05",  0x100000, 0x80000, 0xf4ff18f5 )
	ROM_LOAD16_WORD_SWAP( "sfxj.06a", 0x180000, 0x80000, 0x260d0370 )
	ROM_LOAD16_WORD_SWAP( "sfxj.07",  0x200000, 0x80000, 0x1324d02a )
	ROM_LOAD16_WORD_SWAP( "sfxj.08",  0x280000, 0x80000, 0x2de76f10 )
	ROM_LOAD16_WORD_SWAP( "sfx.09",   0x300000, 0x80000, 0x642fae3f )

	ROM_REGION( 0x1000000, REGION_GFX1, 0 )
	ROMX_LOAD( "ssf.13m",   0x000000, 0x200000, 0xcf94d275 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.15m",   0x000002, 0x200000, 0x5eb703af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.17m",   0x000004, 0x200000, 0xffa60e0f , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.19m",   0x000006, 0x200000, 0x34e825c5 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.14m",   0x800000, 0x100000, 0xb7cc32e7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.16m",   0x800002, 0x100000, 0x8376ad18 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.18m",   0x800004, 0x100000, 0xf5b1b336 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "ssf.20m",   0x800006, 0x100000, 0x459d5c6b , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.21m",   0xc00000, 0x100000, 0xe32854af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.23m",   0xc00002, 0x100000, 0x760f2927 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.25m",   0xc00004, 0x100000, 0x1ee90208 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "sfx.27m",   0xc00006, 0x100000, 0xf814400f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfx.01",   0x00000, 0x08000, 0xb47b8835 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "sfx.02",   0x28000, 0x20000, 0x0022633f )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "sfx.11m",   0x000000, 0x200000, 0x9bdbd476 )
	ROM_LOAD16_WORD_SWAP( "sfx.12m",   0x200000, 0x200000, 0xa05e3aab )
ROM_END

ROM_START( vhunt2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vh2j.03a", 0x000000, 0x80000, 0x9ae8f186 )
	ROM_LOAD16_WORD_SWAP( "vh2j.04a", 0x080000, 0x80000, 0xe2fabf53 )
	ROM_LOAD16_WORD_SWAP( "vh2j.05",  0x100000, 0x80000, 0xde34f624 )
	ROM_LOAD16_WORD_SWAP( "vh2j.06",  0x180000, 0x80000, 0x6a3b9897 )
	ROM_LOAD16_WORD_SWAP( "vh2j.07",  0x200000, 0x80000, 0xb021c029 )
	ROM_LOAD16_WORD_SWAP( "vh2j.08",  0x280000, 0x80000, 0xac873dff )
	ROM_LOAD16_WORD_SWAP( "vh2j.09",  0x300000, 0x80000, 0xeaefce9c )
	ROM_LOAD16_WORD_SWAP( "vh2j.10",  0x380000, 0x80000, 0x11730952 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vh2.13m",   0x0000000, 0x400000, 0x3b02ddaa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.15m",   0x0000002, 0x400000, 0x4e40de66 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.17m",   0x0000004, 0x400000, 0xb31d00c9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.19m",   0x0000006, 0x400000, 0x149be3ab , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.14m",   0x1000000, 0x400000, 0xcd09bd63 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.16m",   0x1000002, 0x400000, 0xe0182c15 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.18m",   0x1000004, 0x400000, 0x778dc4f6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.20m",   0x1000006, 0x400000, 0x605d9d1d , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vh2.01",  0x00000, 0x08000, 0x67b9f779 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vh2.02",  0x28000, 0x20000, 0xaaf15fcb )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vh2.11m",  0x000000, 0x400000, 0x38922efd )
	ROM_LOAD16_WORD_SWAP( "vh2.12m",  0x400000, 0x400000, 0x6e2430af )
ROM_END

ROM_START( vhunt2r1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vh2j.03", 0x000000, 0x80000, 0x1a5feb13 )
	ROM_LOAD16_WORD_SWAP( "vh2j.04", 0x080000, 0x80000, 0x434611a5 )
	ROM_LOAD16_WORD_SWAP( "vh2j.05", 0x100000, 0x80000, 0xde34f624 )
	ROM_LOAD16_WORD_SWAP( "vh2j.06", 0x180000, 0x80000, 0x6a3b9897 )
	ROM_LOAD16_WORD_SWAP( "vh2j.07", 0x200000, 0x80000, 0xb021c029 )
	ROM_LOAD16_WORD_SWAP( "vh2j.08", 0x280000, 0x80000, 0xac873dff )
	ROM_LOAD16_WORD_SWAP( "vh2j.09", 0x300000, 0x80000, 0xeaefce9c )
	ROM_LOAD16_WORD_SWAP( "vh2j.10", 0x380000, 0x80000, 0x11730952 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vh2.13m",   0x0000000, 0x400000, 0x3b02ddaa , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.15m",   0x0000002, 0x400000, 0x4e40de66 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.17m",   0x0000004, 0x400000, 0xb31d00c9 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.19m",   0x0000006, 0x400000, 0x149be3ab , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.14m",   0x1000000, 0x400000, 0xcd09bd63 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.16m",   0x1000002, 0x400000, 0xe0182c15 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.18m",   0x1000004, 0x400000, 0x778dc4f6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vh2.20m",   0x1000006, 0x400000, 0x605d9d1d , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vh2.01",  0x00000, 0x08000, 0x67b9f779 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vh2.02",  0x28000, 0x20000, 0xaaf15fcb )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vh2.11m",  0x000000, 0x400000, 0x38922efd )
	ROM_LOAD16_WORD_SWAP( "vh2.12m",  0x400000, 0x400000, 0x6e2430af )
ROM_END

ROM_START( vsav )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vm3e.03d", 0x000000, 0x80000, 0xf5962a8c )
	ROM_LOAD16_WORD_SWAP( "vm3e.04d", 0x080000, 0x80000, 0x21b40ea2 )
	ROM_LOAD16_WORD_SWAP( "vm3.05a",  0x100000, 0x80000, 0x4118e00f )
	ROM_LOAD16_WORD_SWAP( "vm3.06a",  0x180000, 0x80000, 0x2f4fd3a9 )
	ROM_LOAD16_WORD_SWAP( "vm3.07b",  0x200000, 0x80000, 0xcbda91b8 )
	ROM_LOAD16_WORD_SWAP( "vm3.08a",  0x280000, 0x80000, 0x6ca47259 )
	ROM_LOAD16_WORD_SWAP( "vm3.09b",  0x300000, 0x80000, 0xf4a339e3 )
	ROM_LOAD16_WORD_SWAP( "vm3.10b",  0x380000, 0x80000, 0xfffbb5b8 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vm3.13m",   0x0000000, 0x400000, 0xfd8a11eb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.15m",   0x0000002, 0x400000, 0xdd1e7d4e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.17m",   0x0000004, 0x400000, 0x6b89445e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.19m",   0x0000006, 0x400000, 0x3830fdc7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.14m",   0x1000000, 0x400000, 0xc1a28e6c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.16m",   0x1000002, 0x400000, 0x194a7304 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.18m",   0x1000004, 0x400000, 0xdf9a9f47 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.20m",   0x1000006, 0x400000, 0xc22fc3d9 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vm3.01",   0x00000, 0x08000, 0xf778769b )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vm3.02",   0x28000, 0x20000, 0xcc09faa1 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vm3.11m",   0x000000, 0x400000, 0xe80e956e )
	ROM_LOAD16_WORD_SWAP( "vm3.12m",   0x400000, 0x400000, 0x9cd71557 )
ROM_END

ROM_START( vsavu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vm3u.03d", 0x000000, 0x80000, 0x1f295274 )
	ROM_LOAD16_WORD_SWAP( "vm3u.04d", 0x080000, 0x80000, 0xc46adf81 )
	ROM_LOAD16_WORD_SWAP( "vm3.05a",  0x100000, 0x80000, 0x4118e00f )
	ROM_LOAD16_WORD_SWAP( "vm3.06a",  0x180000, 0x80000, 0x2f4fd3a9 )
	ROM_LOAD16_WORD_SWAP( "vm3.07b",  0x200000, 0x80000, 0xcbda91b8 )
	ROM_LOAD16_WORD_SWAP( "vm3.08a",  0x280000, 0x80000, 0x6ca47259 )
	ROM_LOAD16_WORD_SWAP( "vm3.09b",  0x300000, 0x80000, 0xf4a339e3 )
	ROM_LOAD16_WORD_SWAP( "vm3.10b",  0x380000, 0x80000, 0xfffbb5b8 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vm3.13m",   0x0000000, 0x400000, 0xfd8a11eb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.15m",   0x0000002, 0x400000, 0xdd1e7d4e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.17m",   0x0000004, 0x400000, 0x6b89445e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.19m",   0x0000006, 0x400000, 0x3830fdc7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.14m",   0x1000000, 0x400000, 0xc1a28e6c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.16m",   0x1000002, 0x400000, 0x194a7304 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.18m",   0x1000004, 0x400000, 0xdf9a9f47 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.20m",   0x1000006, 0x400000, 0xc22fc3d9 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vm3.01",   0x00000, 0x08000, 0xf778769b )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vm3.02",   0x28000, 0x20000, 0xcc09faa1 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vm3.11m",   0x000000, 0x400000, 0xe80e956e )
	ROM_LOAD16_WORD_SWAP( "vm3.12m",   0x400000, 0x400000, 0x9cd71557 )
ROM_END

ROM_START( vsavj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vm3j.03d", 0x000000, 0x80000, 0x2a2e74a4 )
	ROM_LOAD16_WORD_SWAP( "vm3j.04d", 0x080000, 0x80000, 0x1c2427bc )
	ROM_LOAD16_WORD_SWAP( "vm3j.05a", 0x100000, 0x80000, 0x95ce88d5 )
	ROM_LOAD16_WORD_SWAP( "vm3j.06b", 0x180000, 0x80000, 0x2c4297e0 )
	ROM_LOAD16_WORD_SWAP( "vm3j.07b", 0x200000, 0x80000, 0xa38aaae7 )
	ROM_LOAD16_WORD_SWAP( "vm3j.08a", 0x280000, 0x80000, 0x5773e5c9 )
	ROM_LOAD16_WORD_SWAP( "vm3j.09b", 0x300000, 0x80000, 0xd064f8b9 )
	ROM_LOAD16_WORD_SWAP( "vm3j.10b", 0x380000, 0x80000, 0x434518e9 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vm3.13m",   0x0000000, 0x400000, 0xfd8a11eb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.15m",   0x0000002, 0x400000, 0xdd1e7d4e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.17m",   0x0000004, 0x400000, 0x6b89445e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.19m",   0x0000006, 0x400000, 0x3830fdc7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.14m",   0x1000000, 0x400000, 0xc1a28e6c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.16m",   0x1000002, 0x400000, 0x194a7304 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.18m",   0x1000004, 0x400000, 0xdf9a9f47 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.20m",   0x1000006, 0x400000, 0xc22fc3d9 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vm3.01",   0x00000, 0x08000, 0xf778769b )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vm3.02",   0x28000, 0x20000, 0xcc09faa1 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vm3.11m",   0x000000, 0x400000, 0xe80e956e )
	ROM_LOAD16_WORD_SWAP( "vm3.12m",   0x400000, 0x400000, 0x9cd71557 )
ROM_END

ROM_START( vsava )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vm3a.03d", 0x000000, 0x80000, 0x44c1198f )
	ROM_LOAD16_WORD_SWAP( "vm3a.04d", 0x080000, 0x80000, 0x2218b781 )
	ROM_LOAD16_WORD_SWAP( "vm3.05a",  0x100000, 0x80000, 0x4118e00f )
	ROM_LOAD16_WORD_SWAP( "vm3.06a",  0x180000, 0x80000, 0x2f4fd3a9 )
	ROM_LOAD16_WORD_SWAP( "vm3.07b",  0x200000, 0x80000, 0xcbda91b8 )
	ROM_LOAD16_WORD_SWAP( "vm3.08a",  0x280000, 0x80000, 0x6ca47259 )
	ROM_LOAD16_WORD_SWAP( "vm3.09b",  0x300000, 0x80000, 0xf4a339e3 )
	ROM_LOAD16_WORD_SWAP( "vm3.10b",  0x380000, 0x80000, 0xfffbb5b8 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vm3.13m",   0x0000000, 0x400000, 0xfd8a11eb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.15m",   0x0000002, 0x400000, 0xdd1e7d4e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.17m",   0x0000004, 0x400000, 0x6b89445e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.19m",   0x0000006, 0x400000, 0x3830fdc7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.14m",   0x1000000, 0x400000, 0xc1a28e6c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.16m",   0x1000002, 0x400000, 0x194a7304 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.18m",   0x1000004, 0x400000, 0xdf9a9f47 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.20m",   0x1000006, 0x400000, 0xc22fc3d9 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vm3.01",   0x00000, 0x08000, 0xf778769b )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vm3.02",   0x28000, 0x20000, 0xcc09faa1 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vm3.11m",   0x000000, 0x400000, 0xe80e956e )
	ROM_LOAD16_WORD_SWAP( "vm3.12m",   0x400000, 0x400000, 0x9cd71557 )
ROM_END

ROM_START( vsavh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vm3h.03a", 0x000000, 0x80000, 0x7cc62df8 )
	ROM_LOAD16_WORD_SWAP( "vm3h.04d", 0x080000, 0x80000, 0xd716f3b5 )
	ROM_LOAD16_WORD_SWAP( "vm3.05a",  0x100000, 0x80000, 0x4118e00f )
	ROM_LOAD16_WORD_SWAP( "vm3.06a",  0x180000, 0x80000, 0x2f4fd3a9 )
	ROM_LOAD16_WORD_SWAP( "vm3.07b",  0x200000, 0x80000, 0xcbda91b8 )
	ROM_LOAD16_WORD_SWAP( "vm3.08a",  0x280000, 0x80000, 0x6ca47259 )
	ROM_LOAD16_WORD_SWAP( "vm3.09b",  0x300000, 0x80000, 0xf4a339e3 )
	ROM_LOAD16_WORD_SWAP( "vm3.10b",  0x380000, 0x80000, 0xfffbb5b8 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vm3.13m",   0x0000000, 0x400000, 0xfd8a11eb , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.15m",   0x0000002, 0x400000, 0xdd1e7d4e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.17m",   0x0000004, 0x400000, 0x6b89445e , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.19m",   0x0000006, 0x400000, 0x3830fdc7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.14m",   0x1000000, 0x400000, 0xc1a28e6c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.16m",   0x1000002, 0x400000, 0x194a7304 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.18m",   0x1000004, 0x400000, 0xdf9a9f47 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vm3.20m",   0x1000006, 0x400000, 0xc22fc3d9 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vm3.01",   0x00000, 0x08000, 0xf778769b )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vm3.02",   0x28000, 0x20000, 0xcc09faa1 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vm3.11m",   0x000000, 0x400000, 0xe80e956e )
	ROM_LOAD16_WORD_SWAP( "vm3.12m",   0x400000, 0x400000, 0x9cd71557 )
ROM_END

ROM_START( vsav2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "vs2j.03", 0x000000, 0x80000, 0x89fd86b4 )
	ROM_LOAD16_WORD_SWAP( "vs2j.04", 0x080000, 0x80000, 0x107c091b )
	ROM_LOAD16_WORD_SWAP( "vs2j.05", 0x100000, 0x80000, 0x61979638 )
	ROM_LOAD16_WORD_SWAP( "vs2j.06", 0x180000, 0x80000, 0xf37c5bc2 )
	ROM_LOAD16_WORD_SWAP( "vs2j.07", 0x200000, 0x80000, 0x8f885809 )
	ROM_LOAD16_WORD_SWAP( "vs2j.08", 0x280000, 0x80000, 0x2018c120 )
	ROM_LOAD16_WORD_SWAP( "vs2j.09", 0x300000, 0x80000, 0xfac3c217 )
	ROM_LOAD16_WORD_SWAP( "vs2j.10", 0x380000, 0x80000, 0xeb490213 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "vs2.13m",   0x0000000, 0x400000, 0x5c852f52 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vs2.15m",   0x0000002, 0x400000, 0xa20f58af , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vs2.17m",   0x0000004, 0x400000, 0x39db59ad , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vs2.19m",   0x0000006, 0x400000, 0x00c763a7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vs2.14m",   0x1000000, 0x400000, 0xcd09bd63 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vs2.16m",   0x1000002, 0x400000, 0xe0182c15 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vs2.18m",   0x1000004, 0x400000, 0x778dc4f6 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "vs2.20m",   0x1000006, 0x400000, 0x605d9d1d , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "vs2.01",   0x00000, 0x08000, 0x35190139 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "vs2.02",   0x28000, 0x20000, 0xc32dba09 )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "vs2.11m",   0x000000, 0x400000, 0xd67e47b7 )
	ROM_LOAD16_WORD_SWAP( "vs2.12m",   0x400000, 0x400000, 0x6d020a14 )
ROM_END

ROM_START( xmcota )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xmne.03e", 0x000000, 0x80000, 0xa9a09b09 )
	ROM_LOAD16_WORD_SWAP( "xmne.04e", 0x080000, 0x80000, 0x52fa2106 )
	ROM_LOAD16_WORD_SWAP( "xmn.05a",  0x100000, 0x80000, 0xac0d7759 )
	ROM_LOAD16_WORD_SWAP( "xmn.06a",  0x180000, 0x80000, 0x1b86a328 )
	ROM_LOAD16_WORD_SWAP( "xmn.07a",  0x200000, 0x80000, 0x2c142a44 )
	ROM_LOAD16_WORD_SWAP( "xmn.08a",  0x280000, 0x80000, 0xf712d44f )
	ROM_LOAD16_WORD_SWAP( "xmn.09a",  0x300000, 0x80000, 0x9241cae8 )
	ROM_LOAD16_WORD_SWAP( "xmn.10a",  0x380000, 0x80000, 0x53c0eab9 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xmn.13m",   0x0000000, 0x400000, 0xbf4df073 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.15m",   0x0000002, 0x400000, 0x4d7e4cef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.17m",   0x0000004, 0x400000, 0x513eea17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.19m",   0x0000006, 0x400000, 0xd23897fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.14m",   0x1000000, 0x400000, 0x778237b7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.16m",   0x1000002, 0x400000, 0x67b36948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.18m",   0x1000004, 0x400000, 0x015a7c4c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.20m",   0x1000006, 0x400000, 0x9dde2758 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xmn.01a",  0x00000, 0x08000, 0x40f479ea )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xmn.02a",  0x28000, 0x20000, 0x39d9b5ad )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xmn.11m",   0x000000, 0x200000, 0xc848a6bc )
	ROM_LOAD16_WORD_SWAP( "xmn.12m",   0x200000, 0x200000, 0x729c188f )
ROM_END

ROM_START( xmcotau )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xmnu.03e", 0x000000, 0x80000, 0x0bafeb0e )
	ROM_LOAD16_WORD_SWAP( "xmnu.04e", 0x080000, 0x80000, 0xc29bdae3 )
	ROM_LOAD16_WORD_SWAP( "xmn.05a",  0x100000, 0x80000, 0xac0d7759 )
	ROM_LOAD16_WORD_SWAP( "xmn.06a",  0x180000, 0x80000, 0x1b86a328 )
	ROM_LOAD16_WORD_SWAP( "xmn.07a",  0x200000, 0x80000, 0x2c142a44 )
	ROM_LOAD16_WORD_SWAP( "xmn.08a",  0x280000, 0x80000, 0xf712d44f )
	ROM_LOAD16_WORD_SWAP( "xmn.09a",  0x300000, 0x80000, 0x9241cae8 )
	ROM_LOAD16_WORD_SWAP( "xmn.10a",  0x380000, 0x80000, 0x53c0eab9 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xmn.13m",   0x0000000, 0x400000, 0xbf4df073 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.15m",   0x0000002, 0x400000, 0x4d7e4cef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.17m",   0x0000004, 0x400000, 0x513eea17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.19m",   0x0000006, 0x400000, 0xd23897fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.14m",   0x1000000, 0x400000, 0x778237b7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.16m",   0x1000002, 0x400000, 0x67b36948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.18m",   0x1000004, 0x400000, 0x015a7c4c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.20m",   0x1000006, 0x400000, 0x9dde2758 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xmn.01a",  0x00000, 0x08000, 0x40f479ea )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xmn.02a",  0x28000, 0x20000, 0x39d9b5ad )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xmn.11m",   0x000000, 0x200000, 0xc848a6bc )
	ROM_LOAD16_WORD_SWAP( "xmn.12m",   0x200000, 0x200000, 0x729c188f )
ROM_END

ROM_START( xmcotah )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xmnh.03", 0x000000, 0x80000, 0xe4b85a90 )
	ROM_LOAD16_WORD_SWAP( "xmnh.04", 0x080000, 0x80000, 0x7dfe1406 )
	ROM_LOAD16_WORD_SWAP( "xmnh.05", 0x100000, 0x80000, 0x87b0ed0f )
	ROM_LOAD16_WORD_SWAP( "xmn.06a", 0x180000, 0x80000, 0x1b86a328 )
	ROM_LOAD16_WORD_SWAP( "xmn.07a", 0x200000, 0x80000, 0x2c142a44 )
	ROM_LOAD16_WORD_SWAP( "xmn.08a", 0x280000, 0x80000, 0xf712d44f )
	ROM_LOAD16_WORD_SWAP( "xmn.09a", 0x300000, 0x80000, 0x9241cae8 )
	ROM_LOAD16_WORD_SWAP( "xmnh.10", 0x380000, 0x80000, 0xcb36b0a4 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xmn.13m",   0x0000000, 0x400000, 0xbf4df073 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.15m",   0x0000002, 0x400000, 0x4d7e4cef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.17m",   0x0000004, 0x400000, 0x513eea17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.19m",   0x0000006, 0x400000, 0xd23897fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.14m",   0x1000000, 0x400000, 0x778237b7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.16m",   0x1000002, 0x400000, 0x67b36948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.18m",   0x1000004, 0x400000, 0x015a7c4c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.20m",   0x1000006, 0x400000, 0x9dde2758 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xmn.01a",  0x00000, 0x08000, 0x40f479ea )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xmn.02a",  0x28000, 0x20000, 0x39d9b5ad )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xmn.11m",   0x000000, 0x200000, 0xc848a6bc )
	ROM_LOAD16_WORD_SWAP( "xmn.12m",   0x200000, 0x200000, 0x729c188f )
ROM_END

ROM_START( xmcotaj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xmnj.03d", 0x000000, 0x80000, 0x79086d62 )
	ROM_LOAD16_WORD_SWAP( "xmnj.04d", 0x080000, 0x80000, 0x38eed613 )
	ROM_LOAD16_WORD_SWAP( "xmn.05",   0x100000, 0x80000, 0xc3ed62a2 )
	ROM_LOAD16_WORD_SWAP( "xmn.06",   0x180000, 0x80000, 0xf03c52e1 )
	ROM_LOAD16_WORD_SWAP( "xmn.07",   0x200000, 0x80000, 0x325626b1 )
	ROM_LOAD16_WORD_SWAP( "xmn.08",   0x280000, 0x80000, 0x7194ea10 )
	ROM_LOAD16_WORD_SWAP( "xmn.09",   0x300000, 0x80000, 0xae946df3 )
	ROM_LOAD16_WORD_SWAP( "xmn.10",   0x380000, 0x80000, 0x32a6be1d )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xmn.13m",   0x0000000, 0x400000, 0xbf4df073 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.15m",   0x0000002, 0x400000, 0x4d7e4cef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.17m",   0x0000004, 0x400000, 0x513eea17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.19m",   0x0000006, 0x400000, 0xd23897fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.14m",   0x1000000, 0x400000, 0x778237b7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.16m",   0x1000002, 0x400000, 0x67b36948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.18m",   0x1000004, 0x400000, 0x015a7c4c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.20m",   0x1000006, 0x400000, 0x9dde2758 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xmn.01a",  0x00000, 0x08000, 0x40f479ea )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xmn.02a",  0x28000, 0x20000, 0x39d9b5ad )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xmn.11m",   0x000000, 0x200000, 0xc848a6bc )
	ROM_LOAD16_WORD_SWAP( "xmn.12m",   0x200000, 0x200000, 0x729c188f )
ROM_END

ROM_START( xmcotaj1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xmnj.03b", 0x000000, 0x80000, 0xc8175fb3 )
	ROM_LOAD16_WORD_SWAP( "xmnj.04b", 0x080000, 0x80000, 0x54b3fba3 )
	ROM_LOAD16_WORD_SWAP( "xmn.05",   0x100000, 0x80000, 0xc3ed62a2 )
	ROM_LOAD16_WORD_SWAP( "xmn.06",   0x180000, 0x80000, 0xf03c52e1 )
	ROM_LOAD16_WORD_SWAP( "xmn.07",   0x200000, 0x80000, 0x325626b1 )
	ROM_LOAD16_WORD_SWAP( "xmn.08",   0x280000, 0x80000, 0x7194ea10 )
	ROM_LOAD16_WORD_SWAP( "xmn.09",   0x300000, 0x80000, 0xae946df3 )
	ROM_LOAD16_WORD_SWAP( "xmn.10",   0x380000, 0x80000, 0x32a6be1d )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xmn.13m",   0x0000000, 0x400000, 0xbf4df073 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.15m",   0x0000002, 0x400000, 0x4d7e4cef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.17m",   0x0000004, 0x400000, 0x513eea17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.19m",   0x0000006, 0x400000, 0xd23897fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.14m",   0x1000000, 0x400000, 0x778237b7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.16m",   0x1000002, 0x400000, 0x67b36948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.18m",   0x1000004, 0x400000, 0x015a7c4c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.20m",   0x1000006, 0x400000, 0x9dde2758 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xmn.01a",  0x00000, 0x08000, 0x40f479ea )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xmn.02a",  0x28000, 0x20000, 0x39d9b5ad )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xmn.11m",   0x000000, 0x200000, 0xc848a6bc )
	ROM_LOAD16_WORD_SWAP( "xmn.12m",   0x200000, 0x200000, 0x729c188f )
ROM_END

ROM_START( xmcotaj2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xmnj.03a", 0x000000, 0x80000, 0x00761611 )
	ROM_LOAD16_WORD_SWAP( "xmnj.04a", 0x080000, 0x80000, 0x614d3f60 )
	ROM_LOAD16_WORD_SWAP( "xmn.05",   0x100000, 0x80000, 0xc3ed62a2 )
	ROM_LOAD16_WORD_SWAP( "xmn.06",   0x180000, 0x80000, 0xf03c52e1 )
	ROM_LOAD16_WORD_SWAP( "xmn.07",   0x200000, 0x80000, 0x325626b1 )
	ROM_LOAD16_WORD_SWAP( "xmn.08",   0x280000, 0x80000, 0x7194ea10 )
	ROM_LOAD16_WORD_SWAP( "xmn.09",   0x300000, 0x80000, 0xae946df3 )
	ROM_LOAD16_WORD_SWAP( "xmn.10",   0x380000, 0x80000, 0x32a6be1d )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xmn.13m",   0x0000000, 0x400000, 0xbf4df073 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.15m",   0x0000002, 0x400000, 0x4d7e4cef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.17m",   0x0000004, 0x400000, 0x513eea17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.19m",   0x0000006, 0x400000, 0xd23897fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.14m",   0x1000000, 0x400000, 0x778237b7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.16m",   0x1000002, 0x400000, 0x67b36948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.18m",   0x1000004, 0x400000, 0x015a7c4c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.20m",   0x1000006, 0x400000, 0x9dde2758 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xmn.01a",  0x00000, 0x08000, 0x40f479ea )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xmn.02a",  0x28000, 0x20000, 0x39d9b5ad )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xmn.11m",   0x000000, 0x200000, 0xc848a6bc )
	ROM_LOAD16_WORD_SWAP( "xmn.12m",   0x200000, 0x200000, 0x729c188f )
ROM_END

ROM_START( xmcotajr )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xmno.03a", 0x000000, 0x80000, 0x7ab19acf )
	ROM_LOAD16_WORD_SWAP( "xmno.04a", 0x080000, 0x80000, 0x7615dd21 )
	ROM_LOAD16_WORD_SWAP( "xmno.05a", 0x100000, 0x80000, 0x0303d672 )
	ROM_LOAD16_WORD_SWAP( "xmno.06a", 0x180000, 0x80000, 0x332839a5 )
	ROM_LOAD16_WORD_SWAP( "xmno.07",  0x200000, 0x80000, 0x6255e8d5 )
	ROM_LOAD16_WORD_SWAP( "xmno.08",  0x280000, 0x80000, 0xb8ebe77c )
	ROM_LOAD16_WORD_SWAP( "xmno.09",  0x300000, 0x80000, 0x5440d950 )
	ROM_LOAD16_WORD_SWAP( "xmno.10a", 0x380000, 0x80000, 0xb8296966 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xmn.13m",   0x0000000, 0x400000, 0xbf4df073 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.15m",   0x0000002, 0x400000, 0x4d7e4cef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.17m",   0x0000004, 0x400000, 0x513eea17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.19m",   0x0000006, 0x400000, 0xd23897fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.14m",   0x1000000, 0x400000, 0x778237b7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.16m",   0x1000002, 0x400000, 0x67b36948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.18m",   0x1000004, 0x400000, 0x015a7c4c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.20m",   0x1000006, 0x400000, 0x9dde2758 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xmn.01",   0x00000, 0x08000, 0x7178336e )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xmn.02",   0x28000, 0x20000, 0x0ec58501 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xmn.11m",   0x000000, 0x200000, 0xc848a6bc )
	ROM_LOAD16_WORD_SWAP( "xmn.12m",   0x200000, 0x200000, 0x729c188f )
ROM_END

ROM_START( xmcotaa )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xmna.03a", 0x000000, 0x80000, 0x7df8b27e )
	ROM_LOAD16_WORD_SWAP( "xmna.04a", 0x080000, 0x80000, 0xb44e30a7 )
	ROM_LOAD16_WORD_SWAP( "xmn.05",   0x100000, 0x80000, 0xc3ed62a2 )
	ROM_LOAD16_WORD_SWAP( "xmn.06",   0x180000, 0x80000, 0xf03c52e1 )
	ROM_LOAD16_WORD_SWAP( "xmn.07",   0x200000, 0x80000, 0x325626b1 )
	ROM_LOAD16_WORD_SWAP( "xmn.08",   0x280000, 0x80000, 0x7194ea10 )
	ROM_LOAD16_WORD_SWAP( "xmn.09",   0x300000, 0x80000, 0xae946df3 )
	ROM_LOAD16_WORD_SWAP( "xmn.10",   0x380000, 0x80000, 0x32a6be1d )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xmn.13m",   0x0000000, 0x400000, 0xbf4df073 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.15m",   0x0000002, 0x400000, 0x4d7e4cef , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.17m",   0x0000004, 0x400000, 0x513eea17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.19m",   0x0000006, 0x400000, 0xd23897fc , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.14m",   0x1000000, 0x400000, 0x778237b7 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.16m",   0x1000002, 0x400000, 0x67b36948 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.18m",   0x1000004, 0x400000, 0x015a7c4c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xmn.20m",   0x1000006, 0x400000, 0x9dde2758 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xmn.01a",  0x00000, 0x08000, 0x40f479ea )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xmn.02a",  0x28000, 0x20000, 0x39d9b5ad )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xmn.11m",   0x000000, 0x200000, 0xc848a6bc )
	ROM_LOAD16_WORD_SWAP( "xmn.12m",   0x200000, 0x200000, 0x729c188f )
ROM_END

ROM_START( xmvsf )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvse.03f", 0x000000, 0x80000, 0xdb06413f )
	ROM_LOAD16_WORD_SWAP( "xvse.04f", 0x080000, 0x80000, 0xef015aef )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvse.03d", 0x000000, 0x80000, 0x5ae5bd3b )
	ROM_LOAD16_WORD_SWAP( "xvse.04d", 0x080000, 0x80000, 0x5eb9c02e )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfu )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvsu.03k", 0x000000, 0x80000, 0x8739ef61 )
	ROM_LOAD16_WORD_SWAP( "xvsu.04k", 0x080000, 0x80000, 0xe11d35c1 )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfur1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvsu.03h", 0x000000, 0x80000, 0x5481155a )
	ROM_LOAD16_WORD_SWAP( "xvsu.04h", 0x080000, 0x80000, 0x1e236388 )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfj )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvsj.03i", 0x000000, 0x80000, 0xef24da96 )
	ROM_LOAD16_WORD_SWAP( "xvsj.04i", 0x080000, 0x80000, 0x70a59b35 )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfjr1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvsj.03d", 0x000000, 0x80000, 0xbeb81de9 )
	ROM_LOAD16_WORD_SWAP( "xvsj.04d", 0x080000, 0x80000, 0x23d11271 )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfjr2 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvsj.03c", 0x000000, 0x80000, 0x180656a1 )
	ROM_LOAD16_WORD_SWAP( "xvsj.04c", 0x080000, 0x80000, 0x5832811c )
	ROM_LOAD16_WORD_SWAP( "xvs.05",   0x100000, 0x80000, 0x030e0e1e )
	ROM_LOAD16_WORD_SWAP( "xvs.06",   0x180000, 0x80000, 0x5d04a8ff )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfa )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvsa.03k", 0x000000, 0x80000, 0xd0cca7a8 )
	ROM_LOAD16_WORD_SWAP( "xvsa.04k", 0x080000, 0x80000, 0x8c8e76fd )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfar1 )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvsa.03e", 0x000000, 0x80000, 0x9bdde21c )
	ROM_LOAD16_WORD_SWAP( "xvsa.04e", 0x080000, 0x80000, 0x33300edf )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfh )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvsh.03a", 0x000000, 0x80000, 0xd4fffb04 )
	ROM_LOAD16_WORD_SWAP( "xvsh.04a", 0x080000, 0x80000, 0x1b4ea638 )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( xmvsfb )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "xvsb.03h", 0x000000, 0x80000, 0x05baccca )
	ROM_LOAD16_WORD_SWAP( "xvsb.04h", 0x080000, 0x80000, 0xe350c755 )
	ROM_LOAD16_WORD_SWAP( "xvs.05a",  0x100000, 0x80000, 0x7db6025d )
	ROM_LOAD16_WORD_SWAP( "xvs.06a",  0x180000, 0x80000, 0xe8e2c75c )
	ROM_LOAD16_WORD_SWAP( "xvs.07",   0x200000, 0x80000, 0x08f0abed )
	ROM_LOAD16_WORD_SWAP( "xvs.08",   0x280000, 0x80000, 0x81929675 )
	ROM_LOAD16_WORD_SWAP( "xvs.09",   0x300000, 0x80000, 0x9641f36b )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "xvs.13m",   0x0000000, 0x400000, 0xf6684efd , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.15m",   0x0000002, 0x400000, 0x29109221 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.17m",   0x0000004, 0x400000, 0x92db3474 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.19m",   0x0000006, 0x400000, 0x3733473c , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.14m",   0x1000000, 0x400000, 0xbcac2e41 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.16m",   0x1000002, 0x400000, 0xea04a272 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.18m",   0x1000004, 0x400000, 0xb0def86a , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "xvs.20m",   0x1000006, 0x400000, 0x4b40ff9f , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "xvs.01",   0x00000, 0x08000, 0x3999e93a )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "xvs.02",   0x28000, 0x20000, 0x101bdee9 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "xvs.11m",   0x000000, 0x200000, 0x9cadcdbc )
	ROM_LOAD16_WORD_SWAP( "xvs.12m",   0x200000, 0x200000, 0x7b11e460 )
ROM_END

ROM_START( hsf2d )
	ROM_REGION( CODE_SIZE, REGION_CPU1, 0 )      /* 68000 code */
	ROM_LOAD16_WORD_SWAP( "hs2a.03", 0x000000, 0x80000, 0xFAEA19DC )
	ROM_LOAD16_WORD_SWAP( "hs2a.04", 0x080000, 0x80000, 0xA956877D )
	ROM_LOAD16_WORD_SWAP( "hs2.05",  0x100000, 0x80000, 0xdde34a35 )
	ROM_LOAD16_WORD_SWAP( "hs2.06",  0x180000, 0x80000, 0xf4e56dda )
	ROM_LOAD16_WORD_SWAP( "hs2a.07", 0x200000, 0x80000, 0xee4420fc )
	ROM_LOAD16_WORD_SWAP( "hs2.08",  0x280000, 0x80000, 0xc9441533 )
	ROM_LOAD16_WORD_SWAP( "hs2.09",  0x300000, 0x80000, 0x3fc638a8 )
	ROM_LOAD16_WORD_SWAP( "hs2.10",  0x380000, 0x80000, 0x20d0f9e4 )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROMX_LOAD( "hs2.13m",   0x0000000, 0x800000, 0xa6ecab17 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "hs2.15m",   0x0000002, 0x800000, 0x10a0ae4d , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "hs2.17m",   0x0000004, 0x800000, 0xadfa7726 , ROM_GROUPWORD | ROM_SKIP(6) )
	ROMX_LOAD( "hs2.19m",   0x0000006, 0x800000, 0xbb3ae322 , ROM_GROUPWORD | ROM_SKIP(6) )

	ROM_REGION( QSOUND_SIZE, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "hs2.01",   0x00000, 0x08000, 0xc1a13786 )
	ROM_CONTINUE(         0x10000, 0x18000 )
	ROM_LOAD( "hs2.02",   0x28000, 0x20000, 0x2d8794aa )

	ROM_REGION( 0x800000, REGION_SOUND1, 0 ) /* QSound samples */
	ROM_LOAD16_WORD_SWAP( "hs2.11m",   0x000000, 0x800000, 0x0e15c359 )
ROM_END


static DRIVER_INIT ( puzloop2 )
{
	init_cps2();
	install_mem_read16_handler(0, 0x804000, 0x804001, pl2_port_0_word_r);
}


GAME( 1993, ssf2,     0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The New Challengers (World 930911)" )
GAME( 1993, ssf2u,    ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The New Challengers (USA 930911)" )
GAME( 1993, ssf2a,    ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The New Challengers (Asia 931005)" )
GAME( 1993, ssf2ar1,  ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The New Challengers (Asia 930914)" )
GAME( 1993, ssf2j,    ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The New Challengers (Japan 931005)" )
GAME( 1993, ssf2jr1,  ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The New Challengers (Japan 930911)" )
GAME( 1993, ssf2jr2,  ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The New Challengers (Japan 930910)" )
GAME( 1993, ssf2tb,   ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The Tournament Battle (World 931119)" )	// works, but not in tournament mode
GAME( 1993, ssf2tbr1, ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The Tournament Battle (World 930911)" )
GAME( 1993, ssf2tbj,  ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II: The Tournament Battle (Japan 930911)" )
GAME( 1993, ddtod,    0,       cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Tower of Doom (Euro 940412)" )
GAME( 1993, ddtodr1,  ddtod,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Tower of Doom (Euro 940113)" )
GAME( 1993, ddtodu,   ddtod,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Tower of Doom (USA 940125)" )
GAME( 1993, ddtodur1, ddtod,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Tower of Doom (USA 940113)" )
GAME( 1993, ddtodj,   ddtod,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Tower of Doom (Japan 940412)" )
GAME( 1993, ddtodjr1, ddtod,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Tower of Doom (Japan 940113)" )
GAME( 1993, ddtodjr2, ddtod,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Tower of Doom (Japan 940125)" )
GAME( 1993, ddtoda,   ddtod,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Tower of Doom (Asia 940113)" )
GAME( 1993, ddtodh,   ddtod,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Tower of Doom (Hispanic 940125)" )
GAME( 1993, ecofghtr, 0,       cps2, sgemf,   cps2, ROT0,   "Capcom", "Eco Fighters (World 931203)" )
GAME( 1993, ecofghtu, ecofghtr,cps2, sgemf,   cps2, ROT0,   "Capcom", "Eco Fighters (USA 940215)" )
GAME( 1993, ecofgtu1, ecofghtr,cps2, sgemf,   cps2, ROT0,   "Capcom", "Eco Fighters (USA 931203)" )
GAME( 1993, uecology, ecofghtr,cps2, sgemf,   cps2, ROT0,   "Capcom", "Ultimate Ecology (Japan 931203)" )
GAME( 1993, ecofghta, ecofghtr,cps2, sgemf,   cps2, ROT0,   "Capcom", "Eco Fighters (Asia 931203)" )
GAME( 1994, ssf2t,    ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II Turbo (World 940223)" )
GAME( 1994, ssf2ta,   ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II Turbo (Asia 940223)" )
GAME( 1994, ssf2tu,   ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II Turbo (USA 940323)" )
GAME( 1994, ssf2tur1, ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II Turbo (USA 940223)" )
GAME( 1994, ssf2xj,   ssf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Street Fighter II X: Grand Master Challenge (Japan 940223)" )
GAME( 1994, xmcota,   0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men: Children of the Atom (Euro 950105)" )
GAME( 1994, xmcotau,  xmcota,  cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men: Children of the Atom (USA 950105)" )
GAME( 1994, xmcotah,  xmcota,  cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men: Children of the Atom (Hispanic 950331)" )
GAME( 1994, xmcotaj,  xmcota,  cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men: Children of the Atom (Japan 941222)" )
GAME( 1994, xmcotaj1, xmcota,  cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men: Children of the Atom (Japan 941219)" )
GAME( 1994, xmcotaj2, xmcota,  cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men: Children of the Atom (Japan 941217)" )
GAME( 1994, xmcotajr, xmcota,  cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men: Children of the Atom (Japan 941208 rent version)" )
GAME( 1994, xmcotaa,  xmcota,  cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men: Children of the Atom (Asia 941217)" )
GAME( 1994, armwar,   0,       cps2, avsp,    cps2, ROT0,   "Capcom", "Armored Warriors (Euro 941024)" )
GAME( 1994, armwarr1, armwar,  cps2, avsp,    cps2, ROT0,   "Capcom", "Armored Warriors (Euro 941011)" )
GAME( 1994, armwaru,  armwar,  cps2, avsp,    cps2, ROT0,   "Capcom", "Armored Warriors (USA 941024)" )
GAME( 1994, pgear,    armwar,  cps2, avsp,    cps2, ROT0,   "Capcom", "Powered Gear: Strategic Variant Armor Equipment (Japan 941024)" )
GAME( 1994, pgearr1,  armwar,  cps2, avsp,    cps2, ROT0,   "Capcom", "Powered Gear: Strategic Variant Armor Equipment (Japan 940916)" )
GAME( 1994, armwara,  armwar,  cps2, avsp,    cps2, ROT0,   "Capcom", "Armored Warriors (Asia 940920)" )
GAME( 1994, avsp,     0,       cps2, avsp,    cps2, ROT0,   "Capcom", "Alien vs. Predator (Euro 940520)" )
GAME( 1994, avspu,    avsp,    cps2, avsp,    cps2, ROT0,   "Capcom", "Alien vs. Predator (USA 940520)" )
GAME( 1994, avspj,    avsp,    cps2, avsp,    cps2, ROT0,   "Capcom", "Alien vs. Predator (Japan 940520)" )
GAME( 1994, avspa,    avsp,    cps2, avsp,    cps2, ROT0,   "Capcom", "Alien vs. Predator (Asia 940520)" )
GAME( 1994, avsph,    avsp,    cps2, avsp,    cps2, ROT0,   "Capcom", "Alien vs. Predator (Hispanic 940520)" )
GAME( 1994, dstlk,    0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Darkstalkers: The Night Warriors (Euro 940705)" )
GAME( 1994, dstlku,   dstlk,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Darkstalkers: The Night Warriors (USA 940818)" )
GAME( 1994, dstlkur1, dstlk,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Darkstalkers: The Night Warriors (USA 940705)" )
GAME( 1994, dstlka,   dstlk,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Darkstalkers: The Night Warriors (Asia 940705)" )
GAME( 1994, vampj,    dstlk,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire: The Night Warriors (Japan 940705)" )	// one of these two must be a bad dump
GAME( 1994, vampja,   dstlk,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire: The Night Warriors (Japan 940705 alt)" )
GAME( 1994, vampjr1,  dstlk,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire: The Night Warriors (Japan 940630)" )
GAME( 1994, ringdest, 0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Ring of Destruction: Slammasters II (Euro 940902)" )
GAME( 1994, smbomb,   ringdest,cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Muscle Bomber: The International Blowout (Japan 940831)" )
GAME( 1994, smbombr1, ringdest,cps2, ssf2,    cps2, ROT0,   "Capcom", "Super Muscle Bomber: The International Blowout (Japan 940808)" )
GAME( 1995, cybots,   0,       cps2, cybots,  cps2, ROT0,   "Capcom", "Cyberbots: Fullmetal Madness (Euro 950424)" )
GAME( 1995, cybotsu,  cybots,  cps2, cybots,  cps2, ROT0,   "Capcom", "Cyberbots: Fullmetal Madness (USA 950424)" )
GAME( 1995, cybotsj,  cybots,  cps2, cybots,  cps2, ROT0,   "Capcom", "Cyberbots: Fullmetal Madness (Japan 950420)" )
GAME( 1995, msh,      0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes (Euro 951024)" )
GAME( 1995, mshu,     msh,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes (USA 951024)" )
GAME( 1995, mshj,     msh,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes (Japan 951117)" )
GAME( 1995, mshjr1,   msh,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes (Japan 951024)" )
GAME( 1995, msha,     msh,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes (Asia 951024)" )
GAME( 1995, mshh,     msh,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes (Hispanic 951117)" )
GAME( 1995, mshb,     msh,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes (Brazil 951117)" )
GAME( 1995, nwarr,    0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Night Warriors: Darkstalkers' Revenge (Euro 950316)" )
GAME( 1995, nwarru,   nwarr,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Night Warriors: Darkstalkers' Revenge (USA 950406)" )
GAME( 1995, nwarrh,   nwarr,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Night Warriors: Darkstalkers' Revenge (Hispanic 950403)" )
GAME( 1995, nwarrb,   nwarr,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Night Warriors: Darkstalkers' Revenge (Brazil 950403)" )
GAME( 1995, vhuntj,   nwarr,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Hunter: Darkstalkers' Revenge (Japan 950316)" )
GAME( 1995, vhuntjr1, nwarr,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Hunter: Darkstalkers' Revenge (Japan 950307)" )
GAME( 1995, vhuntjr2, nwarr,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Hunter: Darkstalkers' Revenge (Japan 950302)" )
GAME( 1995, sfa,      0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha: Warriors' Dreams (Euro 950727)" )
GAME( 1995, sfar1,    sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha: Warriors' Dreams (Euro 950718)" )
GAME( 1995, sfar2,    sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha: Warriors' Dreams (Euro 950627)" )
GAME( 1995, sfar3,    sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha: Warriors' Dreams (Euro 950605)" )
GAME( 1995, sfau,     sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha: Warriors' Dreams (USA 950627)" )
GAME( 1995, sfza,     sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero (Asia 950627)" )
GAME( 1995, sfzj,     sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero (Japan 950727)" )
GAME( 1995, sfzjr1,   sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero (Japan 950627)" )
GAME( 1995, sfzjr2,   sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero (Japan 950605)" )
GAME( 1995, sfzh,     sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero (Hispanic 950627)" )
GAME( 1995, sfzb,     sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero (Brazil 951109)" )
GAME( 1995, sfzbr1,   sfa,     cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero (Brazil 950727)" )
GAME( 1995, mmancp2u, megaman, cps2, sgemf,   cps2, ROT0,   "Capcom", "Mega Man - The Power Battle (CPS2, USA 951006)" )
GAME( 1995, rmancp2j, megaman, cps2, sgemf,   cps2, ROT0,   "Capcom", "Rockman: The Power Battle (CPS2, Japan 950922)" )
GAME( 1996, 19xx,     0,       cps2, 19xx,    cps2, ROT270, "Capcom", "19XX: The War Against Destiny (USA 951207)" )
GAME( 1996, 19xxa,    19xx,    cps2, 19xx,    cps2, ROT270, "Capcom", "19XX: The War Against Destiny (Asia 951207)" )
GAME( 1996, 19xxj,    19xx,    cps2, 19xx,    cps2, ROT270, "Capcom", "19XX: The War Against Destiny (Japan 951225)" )
GAME( 1996, 19xxjr1,  19xx,    cps2, 19xx,    cps2, ROT270, "Capcom", "19XX: The War Against Destiny (Japan 951207)" )
GAME( 1996, 19xxh,    19xx,    cps2, 19xx,    cps2, ROT270, "Capcom", "19XX: The War Against Destiny (Hispanic 951218)" )
GAME( 1996, 19xxb,    19xx,    cps2, 19xx,    cps2, ROT270, "Capcom", "19XX: The War Against Destiny (Brazil 951218)" )
GAME( 1996, ddsom,    0,       cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (Euro 960619)" )
GAME( 1996, ddsomr1,  ddsom,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (Euro 960223)" )
GAME( 1996, ddsomr2,  ddsom,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (Euro 960209)" )
GAME( 1996, ddsomr3,  ddsom,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (Euro 960208)" )
GAME( 1996, ddsomu,   ddsom,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (USA 960619)" )
GAME( 1996, ddsomur1, ddsom,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (USA 960209)" )
GAME( 1996, ddsomj,   ddsom,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (Japan 960619)" )
GAME( 1996, ddsomjr1, ddsom,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (Japan 960206)" )
GAME( 1996, ddsoma,   ddsom,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (Asia 960619)" )
GAME( 1996, ddsomb,   ddsom,   cps2, ddtod,   cps2, ROT0,   "Capcom", "Dungeons & Dragons: Shadow over Mystara (Brazil 960223)" )
GAME( 1996, megaman2, 0,       cps2, sgemf,   cps2, ROT0,   "Capcom", "Mega Man 2: The Power Fighters (USA 960708)" )
GAME( 1996, megamn2a, megaman2,cps2, sgemf,   cps2, ROT0,   "Capcom", "Mega Man 2: The Power Fighters (Asia 960708)" )
GAME( 1996, rckman2j, megaman2,cps2, sgemf,   cps2, ROT0,   "Capcom", "Rockman 2: The Power Fighters (Japan 960708)" )
GAME( 1996, qndream,  0,       cps2, qndream, cps2, ROT0,   "Capcom", "Quiz Nanairo Dreams: Nijiirochou no Kiseki (Japan 960826)" )
GAME( 1996, sfa2,     0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha 2 (USA 960306)" )
GAME( 1996, sfz2j,    sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 (Japan 960227)" )
GAME( 1996, sfz2a,    sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 (Asia 960227)" )
GAME( 1996, sfz2b,    sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 (Brazil 960531)" )
GAME( 1996, sfz2br1,  sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 (Brazil 960304)" )
GAME( 1996, sfz2h,    sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 (Hispanic 960304)" )
GAME( 1996, sfz2n,    sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 (Oceania 960229)" )
GAME( 1996, sfz2aj,   sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 Alpha (Japan 960805)" )
GAME( 1996, sfz2ah,   sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 Alpha (Hispanic 960813)" )
GAME( 1996, sfz2ab,   sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 Alpha (Brazil 960813)" )
GAME( 1996, sfz2aa,   sfa2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 2 Alpha (Asia 960826)" )
GAME( 1996, spf2t,    0,       cps2, 19xx,    cps2, ROT0,   "Capcom", "Super Puzzle Fighter II Turbo (USA 960620)" )
GAME( 1996, spf2xj,   spf2t,   cps2, 19xx,    cps2, ROT0,   "Capcom", "Super Puzzle Fighter II X (Japan 960531)" )
GAME( 1996, spf2ta,   spf2t,   cps2, 19xx,    cps2, ROT0,   "Capcom", "Super Puzzle Fighter II Turbo (Asia 960529)" )
GAME( 1996, xmvsf,    0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (Euro 961004)" )
GAME( 1996, xmvsfr1,  xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (Euro 960910)" )
GAME( 1996, xmvsfu,   xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (USA 961023)" )
GAME( 1996, xmvsfur1, xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (USA 961004)" )
GAME( 1996, xmvsfj,   xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (Japan 961004)" )
GAME( 1996, xmvsfjr1, xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (Japan 960910)" )
GAME( 1996, xmvsfjr2, xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (Japan 960909)" )
GAME( 1996, xmvsfa,   xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (Asia 961023)" )
GAME( 1996, xmvsfar1, xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (Asia 960919)" )
GAME( 1996, xmvsfh,   xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (Hispanic 961004)" )
GAME( 1996, xmvsfb,   xmvsf,   cps2, ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (Brazil 961023)" )
GAME( 1997, batcir,   0,       cps2, batcir,  cps2, ROT0,   "Capcom", "Battle Circuit (Euro 970319)" )
GAME( 1997, batcira,  batcir,  cps2, batcir,  cps2, ROT0,   "Capcom", "Battle Circuit (Asia 970319)" )
GAME( 1997, batcirj,  batcir,  cps2, batcir,  cps2, ROT0,   "Capcom", "Battle Circuit (Japan 970319)" )
GAME( 1997, csclub,   0,       cps2, sgemf,   cps2, ROT0,   "Capcom", "Capcom Sports Club (Euro 970722)" )
GAME( 1997, cscluba,  csclub,  cps2, sgemf,   cps2, ROT0,   "Capcom", "Capcom Sports Club (Asia 970722)" )
GAME( 1997, csclubj,  csclub,  cps2, sgemf,   cps2, ROT0,   "Capcom", "Capcom Sports Club (Japan 970722)" )
GAME( 1997, csclubh,  csclub,  cps2, sgemf,   cps2, ROT0,   "Capcom", "Capcom Sports Club (Hispanic 970722)" )
GAME( 1997, mshvsf,   0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (Euro 970625)" )
GAME( 1997, mshvsfu,  mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (USA 970827)" )
GAME( 1997, mshvsfu1, mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (USA 970625)" )
GAME( 1997, mshvsfj,  mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (Japan 970707)" )
GAME( 1997, mshvsfj1, mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (Japan 970702)" )
GAME( 1997, mshvsfj2, mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (Japan 970625)" )
GAME( 1997, mshvsfh,  mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (Hispanic 970625)" )
GAME( 1997, mshvsfa,  mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (Asia 970625)" )
GAME( 1997, mshvsfa1, mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (Asia 970620)" )
GAME( 1997, mshvsfb,  mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (Brazil 970827)" )
GAME( 1997, mshvsfb1, mshvsf,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Super Heroes Vs. Street Fighter (Brazil 970625)" )
GAME( 1997, sgemf,    0,       cps2, sgemf,   cps2, ROT0,   "Capcom", "Super Gem Fighter Mini Mix (USA 970904)" )
GAME( 1997, pfghtj,   sgemf,   cps2, sgemf,   cps2, ROT0,   "Capcom", "Pocket Fighter (Japan 970904)" )
GAME( 1997, sgemfa,   sgemf,   cps2, sgemf,   cps2, ROT0,   "Capcom", "Super Gem Fighter: Mini Mix (Asia 970904)" )
GAME( 1997, sgemfh,   sgemf,   cps2, sgemf,   cps2, ROT0,   "Capcom", "Super Gem Fighter: Mini Mix (Hispanic 970904)" )
GAME( 1997, vhunt2,   0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Hunter 2: Darkstalkers Revenge (Japan 970929)" )
GAME( 1997, vhunt2r1, vhunt2,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Hunter 2: Darkstalkers Revenge (Japan 970913)" )
GAME( 1997, vsav,     0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Savior: The Lord of Vampire (Euro 970519)" )
GAME( 1997, vsavu,    vsav,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Savior: The Lord of Vampire (USA 970519)" )
GAME( 1997, vsavj,    vsav,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Savior: The Lord of Vampire (Japan 970519)" )
GAME( 1997, vsava,    vsav,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Savior: The Lord of Vampire (Asia 970519)" )
GAME( 1997, vsavh,    vsav,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Savior: The Lord of Vampire (Hispanic 970519)" )
GAME( 1997, vsav2,    0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Vampire Savior 2: The Lord of Vampire (Japan 970913)" )
GAME( 1998, mvsc,     0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Vs. Capcom: Clash of Super Heroes (Euro 980112)" )
GAME( 1998, mvscu,    mvsc,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Vs. Capcom: Clash of Super Heroes (USA 980123)" )
GAME( 1998, mvscj,    mvsc,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Vs. Capcom: Clash of Super Heroes (Japan 980123)" )
GAME( 1998, mvscjr1,  mvsc,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Vs. Capcom: Clash of Super Heroes (Japan 980112)" )
GAME( 1998, mvsca,    mvsc,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Vs. Capcom: Clash of Super Heroes (Asia 980123)" )
GAME( 1998, mvscar1,  mvsc,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Vs. Capcom: Clash of Super Heroes (Asia 980112)" )
GAME( 1998, mvsch,    mvsc,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Vs. Capcom: Clash of Super Heroes (Hispanic 980123)" )
GAME( 1998, mvscb,    mvsc,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Marvel Vs. Capcom: Clash of Super Heroes (Brazil 980123)" )
GAME( 1998, sfa3,     0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha 3 (Euro 980904)" )
GAME( 1998, sfa3u,    sfa3,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha 3 (USA 980904)" )
GAME( 1998, sfa3ur1,  sfa3,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha 3 (USA 980629)" )
GAME( 1998, sfa3b,    sfa3,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha 3 (Brazil 980629)" )
GAME( 1998, sfz3j,    sfa3,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 3 (Japan 980904)" )
GAME( 1998, sfz3jr1,  sfa3,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 3 (Japan 980727)" )
GAME( 1998, sfz3jr2,  sfa3,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 3 (Japan 980629)" )
GAME( 1998, sfz3a,    sfa3,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 3 (Asia 980904)" )
GAME( 1998, sfz3ar1,  sfa3,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 3 (Asia 980701)" )
GAME( 1999, jyangoku, 0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Jyangokushi: Haoh no Saihai (Japan)" )
GAME( 2004, hsf2,     0,       cps2, ssf2,    cps2, ROT0,   "Capcom", "Hyper Street Fighter II: The Anniversary Edition (Asia 040202)" )
GAME( 2003, hsf2j,    hsf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Hyper Street Fighter II: The Anniversary Edition (Japan 031222)" )
GAME( 2004, hsf2d,    hsf2,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Hyper Street Fighter II: The Anniversary Edition (Asia 040202) Phoenix Edition" )

/* Games released on CPS-2 hardware by Takumi */

GAME( 1999, gigawing, 0,       dima, 19xx,    cps2, ROT0,   "Capcom, supported by Takumi", "Giga Wing (USA 990222)" )
GAME( 1999, gwingj,   gigawing,dima, 19xx,    cps2, ROT0,   "Capcom, supported by Takumi", "Giga Wing (Japan 990223)" )
GAME( 1999, gwinga,   gigawing,dima, 19xx,    cps2, ROT0,   "Capcom, supported by Takumi", "Giga Wing (Asia 990222)" )
GAME( 2000, mmatrix,  0,       dima, 19xx,    cps2, ROT0,   "Capcom, supported by Takumi", "Mars Matrix: Hyper Solid Shooting (USA 000412)" )
GAME( 2000, mmatrixj, mmatrix, dima, 19xx,    cps2, ROT0,   "Capcom, supported by Takumi", "Mars Matrix: Hyper Solid Shooting (Japan 000412)" )

/* Games released on CPS-2 hardware by Mitchell */

GAME( 2000, mpang,    0,       cps2, ssf2,    cps2, ROT0,   "Mitchell, distributed by Capcom", "Mighty! Pang (USA 001010)" )
GAME( 2000, mpangj,   mpang,   cps2, ssf2,    cps2, ROT0,   "Mitchell, distributed by Capcom", "Mighty! Pang (Japan 001011)" )
GAME( 2001, pzloop2,  0,       cps2, puzloop2,puzloop2,ROT0,"Mitchell, distributed by Capcom", "Puzz Loop 2 (Euro 010302)" )
GAME( 2001, pzloop2j, pzloop2, cps2, puzloop2,puzloop2,ROT0,"Mitchell, distributed by Capcom", "Puzz Loop 2 (Japan 010205)" )
GAME( 2001, choko,    0,       cps2, cps2,    cps2, ROT0,   "Mitchell, distributed by Capcom", "Choko (Japan 010820)" )

/* Games released on CPS-2 hardware by Eighting/Raizing */

GAME( 2000, dimahoo,  0,       dima, sgemf,   cps2, ROT270, "Eighting/Raizing, distributed by Capcom", "Dimahoo (Euro 000121)" )
GAME( 2000, dimahoou, dimahoo, cps2, sgemf,   cps2, ROT270, "Eighting/Raizing, distributed by Capcom", "Dimahoo (USA 000121)" )
GAME( 2000, gmahou,   dimahoo, dima, sgemf,   cps2, ROT270, "Eighting/Raizing, distributed by Capcom", "Great Mahou Daisakusen (Japan 000121)" )
GAME( 2000, 1944,     0,       cps2, 19xx,    cps2, ROT0,   "Capcom, supported by Eighting/Raizing", "1944: The Loop Master (USA 000620)" )
GAME( 2000, 1944j,    1944,    cps2, 19xx,    cps2, ROT0,   "Capcom, supported by Eighting/Raizing", "1944: The Loop Master (Japan 000620)" )

/* Games released on CPS-2 hardware by Cave */

GAME( 2001, progear,  0,       cps2, sgemf,   cps2, ROT0,   "Capcom, supported by Cave", "Progear (USA 010117)" )
GAME( 2001, progearj, progear, cps2, sgemf,   cps2, ROT0,   "Capcom, supported by Cave", "Progear no Arashi (Japan 010117)" )
GAME( 2001, progeara, progear, cps2, sgemf,   cps2, ROT0,   "Capcom, supported by Cave", "Progear (Asia 010117)" )

/* region changed version */
GAME_HACK(1996, 19xxregion,     19xx,     cps2,  19xx,    cps2, ROT270, "Capcom", "19XX: The War Against Destiny (region changed to Japan)" )
GAME_HACK(1997, batcirjregion,  batcirj,  cps2,  batcir,  cps2, ROT0,   "Capcom", "Battle Circuit (region changed to USA)" )
GAME_HACK(1993, ecofghtrregion, ecofghtr, cps2,  cps2,    cps2, ROT0,   "Capcom", "Ultimate Ecology (region changed to Japan)" )
GAME_HACK(1996, xmvsfregion,    xmvsfu,   cps2,  ssf2,    cps2, ROT0,   "Capcom", "X-Men Vs. Street Fighter (region changed to Japan)" )

GAME_HACK( 1998, sfa3p,     sfa3u,    cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha 3 (US 980904) (Battle Modes Unlocked)" )
GAME_HACK( 1998, sfa3r1p,   sfa3ur1,  cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Alpha 3 (US 980629) (Battle Modes Unlocked)" )
GAME_HACK( 1998, sfz3jp,    sfz3jr1,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 3 (Japan 980727) (Battle Modes Unlocked)" )
GAME_HACK( 1998, sfz3jr1p,  sfz3jr2, cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 3 (Japan 980629) (Battle Modes Unlocked)" )
GAME_HACK( 1998, sfz3ap,    sfz3a,   cps2, ssf2,    cps2, ROT0,   "Capcom", "Street Fighter Zero 3 (Asia 980701) (Battle Modes Unlocked)" )
